unit XMLUtil;

interface
uses Classes, SysUtils, MSXML2_TLB, DB, FmtBcd;

type
  EXMLParse = class(Exception)
    NativeMessage: string;
    FileName: string;
    LineNo,
    LinePos,
    ErrorCode: integer;
  end;
  EXMLMustGetValue = class(Exception);

  TXMLDocHelper = class(TDOMDocument)
  private
    FValidateOnSave: boolean;
    FDecimalSettings: TFormatSettings;
    function GetValidateOnParse: boolean;
    procedure SetValidateOnParse(const Value: boolean);
    function GetIntf: IXMLDOMDocument2;
    function GetParseError: IXMLDOMParseError;
    {
      NodeByName, InternalSearchForNode -  ,   ,    ( !!!)
      NodeByNameDepth, InternalSearchForNodeDepth -  ,   (   )
    }
    function InternalSearchForNode(RootNode: IXMLDOMNode; NodeName: string;
      MaxDepth, CurDepth: longint; NodeType: DOMNodeType): IXMLDOMNode; deprecated;
    function InternalSearchForNodeDepth(RootNode: IXMLDOMNode; NodeName: string;
      MaxDepth, CurDepth: longint; NodeType: DOMNodeType): IXMLDOMNode;

    function GetABNS(Node: IXMLDOMNode; AttrName: String): String;
    procedure SetABNS(Node: IXMLDOMNode; AttrName: String;
      const Value: String);
    function GetABNI(Node: IXMLDOMNode; AttrName: String): Integer;
    procedure SetABNI(Node: IXMLDOMNode; AttrName: String;
      const Value: Integer);
    function GetRoot: IXMLDOMElement;
    function GetAsString: string;
  public
    constructor Create(AOwner: TComponent); override;
    procedure LoadFromStream(Source: TStream);
    (**
     *  XML  .
     *
     * @param Source   
     * @param RaiseOnParseError     Exception    (  - False)
     *)
    procedure LoadFromFile(Source: String; RaiseOnParseError: Boolean = False);
    procedure LoadFromString(Source: string);

    procedure SaveToStream(Stream: TStream; Overwrite: boolean = True);
    procedure SaveToFile(FileName: string; Overwrite: boolean = True);

    function IsNullDate(RootNode: IXMLDOMNode): boolean;

    function ForceNodes(RootNode: IXMLDOMNode; Nodes2Create: array of string; UseExistsNodes: boolean = False): IXMLDOMNode;
    {
      NodeByName, InternalSearchForNode -  ,   ,    ( !!!)
      NodeByNameDepth, InternalSearchForNodeDepth -  ,   (   )
    }
    function NodeByName(RootNode: IXMLDOMNode; NodeName: string;
                        MaxDepth: longint = 0; includeSelf: boolean = False): IXMLDOMNode; deprecated; //ALE  MaxDepth    0  1   .   ...
                                                                                           //  -  
    function NodeByNameDepth(RootNode: IXMLDOMNode; NodeName: string;
      MaxDepth: longint = 0; includeSelf: boolean = False): IXMLDOMNode;

    function NamedParent(ChildNode: IXMLDOMNode; RootName: string): IXMLDOMNode;
    function AddAttribute(Node: IXMLDOMNode; AttrName: string; AttrValue: string = ''): IXMLDOMAttribute; overload;
    function AddAttribute(Node: IXMLDOMNode; AttrName: string; AttrValue: OleVariant): IXMLDOMAttribute; overload;
    procedure DeleteAttribute(Node: IXMLDOMNode; AttrName: string);
    property ABNS[Node: IXMLDOMNode; AttrName: String]: String read GetABNS write SetABNS;
    property ABNI[Node: IXMLDOMNode; AttrName: String]: Integer read GetABNI write SetABNI;

    property ParseError: IXMLDOMParseError read GetParseError;
    property Intf: IXMLDOMDocument2 read GetIntf;
    property ValidateOnParse: boolean read GetValidateOnParse
                                      write SetValidateOnParse;
    property ValidateOnSave: boolean read FValidateOnSave write FValidateOnSave;
    property Root: IXMLDOMElement read GetRoot;

    function AttrExists(Node: IXMLDOMNode; AttrName: string): boolean;
    function NodeExists(RootNode: IXMLDOMNode; NodeName: string): boolean;
    function AppendSibling(Node: IXMLDOMNode; NodeName: string): IXMLDOMNode;

    (**
     *       XML (..     ..)
     *
     * @param RootNode  ,     XML
     * @param XMLText   XML    
     *)
    procedure AppendXML(RootNode: IXMLDOMNode; XMLText: string); overload;

    (**
     *       XML (..     ..)
     *
     * @param Path        ,     XML
     * @param XMLText   XML    
     *)
    procedure AppendXML(Path: string; XMLText: string); overload;

    //procedure AppendFrom(XMLDoc: TXMLDocHelper; FromNode: IXML);

    //GetValues
    function GetStdDateTime(RootNode: IXMLDOMNode; RaiseOnError:boolean = True): TDateTime;
    function GetText(RootNode: IXMLDOMNode; NodeName: string = '';
          RaiseIfNotFound : boolean = False): string;
    function MustGetText(RootNode: IXMLDOMNode; NodeName: string = ''): string;
    function GetDateTime(RootNode: IXMLDOMNode; NodeName: string): TDateTime;
    function MustGetDateTime(RootNode: IXMLDOMNode; NodeName: string): TDateTime;
    function GetISODate(Node: IXMLDomNode): TDateTime;
    function GetISODateTime(Node: IXMLDomNode): TDateTime;
    function GetISODateOrDateTime(Node: IXMLDomNode; NodeName: string = '';
          RaiseIfNotFound : boolean = False ): TDateTime;
    function GetDecimalNumber(RootNode: IXMLDomNode; NodeName: string = '';
          RaiseIfNotFound : boolean = False ): Currency;
    function GetBCDNumber(RootNode: IXMLDomNode; NodeName: string = '';
          RaiseIfNotFound : boolean = False ): TBCD;
    function GetPartadDate(Node: IXMLDomNode; NodeName: string = '';
          RaiseIfNotFound: boolean = False): TDateTime;

    function GetPartadDateOrDateTime(RootNode: IXMLDomNode; NodeName: string = '';
          RaiseIfNotFound: boolean = False): TDateTime;

    function MustGetDecimalNumber(RootNode: IXMLDOMNode; NodeName: string = ''): Currency;

    procedure AddStdDateTime(RootNode: IXMLDOMNode; DateTime: TDateTime; WriteZeroDates: boolean = True);
    procedure SetISODate(Node: IXMLDomNode; Date: TDateTime);
    procedure SetISODateTime(Node: IXMLDomNode; Date: TDateTime);
    procedure SetISODateTimeEx(Node: IXMLDomNode; Date: TDateTime);
    procedure SetDecimalNumber(Node: IXMLDomNode; Value: Currency);

    procedure PrepareBlank(BaseElement: string; UpCase: boolean = true; isDOCTYPE: boolean = True);

    //AddNodeFromDS -> StoreToDS
    procedure AddNodesFromDS(RootNode: IXMLDOMNode; DS: TDataSet;
       FieldList: array of string; ISExcludeFieldList : boolean;
       UseExistsNodes: boolean);
    procedure ReplDSFromNodes(RootNode: IXMLDOMNode; DS: TDataSet;
       FieldList: array of string; ISExcludeFieldList: boolean; UpCase: boolean = true);

    function TransformWithXSL(const XSLTemplate: string): String; overload;

    procedure TransformWithXSL(const XSLTemplate, FileName: string); overload;
    procedure TransformWithXSL(const XSLTemplate: string; Stream: TStream); overload;
    procedure TransformSelfWithXSL(const XSLTemplate: string);

    function ValidateAgainstExternalXSD(XSDFileName: string): IXMLDOMParseError;

    function ValidateAgainstDTD(DTDFileName: string): IXMLDOMParseError;

    property AsString: string read GetAsString;
    function AppendChild(Root: IXMLDomNode; ChildNodeName: string): IXMLDomNode; overload;
    function AppendChild(Root: IXMLDomNode; ChildNodeName: string; AChildNodeValue: OleVariant): IXMLDomNode; overload;
    function AppendChildNS(Root: IXMLDomNode; ChildNodeName: string; NamespaceURI: string = ''): IXMLDomNode;

    function GetNodePathName(RootNode: IXMLDomNode; NodeName: string):string;
    (**
     *    ,    
     *)
    procedure RemoveEmptyNodes({RootNode: IXMLDomNode});
    procedure RemoveEmptyNodesQuick;

    function IsWellFormedXmlText(AnXmlText: string): boolean;
    function LoadWithoutErrors: boolean;
    function GetNodeXPath(ANode: IXMLDomNode): string;
    (**
     *         
     *)
    function GetNextNode(ANode: IXMLDomNode): IXMLDomNode;
    function GetEncoding: string;
    procedure AddCDataSection(Node: IXMLDomNode; CData: string);
  end;

function DateTimeToISOXML(Date: TDateTime): string;

function InList(const V: variant; const TestVal: array of variant): Boolean;

function WordCount(const S: string; const WordDelims: TSysCharSet): Integer;

function WordPosition(const N: Integer; const S: string;
  const WordDelims: TSysCharSet): Integer;

function ExtractWord(N: Integer; const S: string;
  const WordDelims: TSysCharSet): string;

function XMLDate2DateTime(XMLDate : string; DefDateTime : TDateTime = -700000) : TDateTime;

procedure XMLError(Error: IXMLDOMParseError);

procedure DecorateXML(XmlDocName: string);

procedure CreateXMLFileByDS(const BaseElement: string; DS: TDataSet;
   FieldList: array of string; ISExcludeFieldList: boolean;
   const OutFileName: string );

procedure ReplDSFromXMLFile(const XMLFileName: string; DS: TDataSet);

// 2 XML ,         .
//ExcludeElemant .
function CompareXMLFile( XMLFile1, XMLFile2: string;
    KeyElements:array of string;
    ExcludeElement:array of variant;
    File1Name : string = '';
    File2Name : string = '' ): string;

procedure PreScanXML;
function  ScanXML(var CurrentNode: IXMLDOMNode; ScanNodeName: string): boolean;

function GetRootNodeName(XMLFileName: string): string;
procedure GetXMLRootVersion(XMLFileName: string; var RootName, Version: string);

//procedure AmountInWriting(FileNameIn, FileNameOut: string; NodeNames: array of string);
procedure TransformXmlFile(XmlFileName, XslFileName, OutFileName: string);

procedure RemoveEmptyNodeQuick(const FileName, OutFileName:string);

Function InListUC(v:string; const TestVal:array of string):boolean;


function CreateMSXML2DOMDocument: IDispatch;
function CreateMSXML2XMLSchemaCache: IDispatch;


const
  XMLNS_ATTR_NAME = 'xmlns';

  XMLNS_XSI_ATTR_NAME = 'xmlns:xsi';
  XMLNS_XSI_ATTR_VALUE = 'http://www.w3.org/2001/XMLSchema-instance';
  XMLNS_XSI_ATTR = 'xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"';

  XSI_NONAMESPACE_ATTR_NAME = 'xsi:noNamespaceSchemaLocation';
  XSI_NONAMESPACE_ATTR_VALUE = '%s';
  XSI_NONAMESPACE_ATTR = 'xsi:noNamespaceSchemaLocation="%s"';

implementation

uses ActiveX, ComObj, Dialogs,
  Variants, DateUtils, StrUtils, Math;

var
 _MSXML2_DOMDocument_Version: string;
 _MSXML2_XMLSchemaCache_Version: string;

function DateTimeToISOXML(Date: TDateTime): String;
begin
  Result := FormatDateTime('yyyy"-"mm"-"dd"T"hh":"nn":"ss', Date);
end;

function InList(const V: variant; const TestVal: array of variant): Boolean;
var
  k: Integer;
begin
  for k := 0 to Length(TestVal) - 1 do begin
    if V = TestVal[k] then begin
      Result := True;
      Exit;
    end;
  end;
  Result := False;
end;

function WordCount(const S: string; const WordDelims: TSysCharSet): Integer;
var
  SLen, I: Cardinal;
begin
  Result := 0;
  I := 1;
  SLen := Length(S);
  while I <= SLen do
  begin
    while (I <= SLen) and CharInSet(S[I], WordDelims) do
      Inc(I);
    if I <= SLen then
      Inc(Result);
    while (I <= SLen) and not CharInSet(S[I], WordDelims) do
      Inc(I);
  end;
end;

function WordPosition(const N: Integer; const S: string;
  const WordDelims: TSysCharSet): Integer;
var
  Count, I: Integer;
begin
  Count := 0;
  I := 1;
  Result := 0;
  while (I <= Length(S)) and (Count <> N) do
  begin
    { skip over delimiters }
    while (I <= Length(S)) and CharInSet(S[I], WordDelims) do
      Inc(I);
    { if we're not beyond end of S, we're at the start of a word }
    if I <= Length(S) then
      Inc(Count);
    { if not finished, find the end of the current word }
    if Count <> N then
      while (I <= Length(S)) and not CharInSet(S[I], WordDelims) do
        Inc(I)
    else
      Result := I;
  end;
end;

function ExtractWord(N: Integer; const S: string;
  const WordDelims: TSysCharSet): string;
var
  I: Integer;
  Len: Integer;
begin
  Len := 0;
  I := WordPosition(N, S, WordDelims);
  if I <> 0 then
    { find the end of the current word }
    while (I <= Length(S)) and not CharInSet(S[I], WordDelims) do
    begin
      { add the I'th character to result }
      Inc(Len);
      SetLength(Result, Len);
      Result[Len] := S[I];
      Inc(I);
    end;
  SetLength(Result, Len);
end;

function XMLDate2DateTime(XMLDate: string;
  DefDateTime: TDateTime): TDateTime;
var
  y,m,d : integer;
begin
    y := StrToIntDef(copy(XMLDate,1,4),0);
    m := StrToIntDef(copy(XMLDate,6,2),0);
    d := StrToIntDef(copy(XMLDate,9,2),0);
    if (y <= 0) or (m <= 0) or (d <= 0) then
      result := DefDateTime
    else
      try
        result := EncodeDate(y,m,d);
      except
        result := DefDateTime;
      end;
end;

procedure XMLError(Error: IXMLDOMParseError);
var
  E: EXMLParse;
begin
  E := EXMLParse.CreateFmt('  XML :'#13+
                           '  %x'#13+
                           ' "%s"'#13+
                           ' "%s"'#13+
                           ' %d'#13+
                           ' %d'#13,
                           [Error.Get_ErrorCode,
                            Error.get_Reason,
                            Error.get_url,
                            Error.Get_Line,
                            Error.Get_LinePos]);

  E.NativeMessage := Error.reason;
  E.LineNo := Error.Get_line;
  E.LinePos := Error.Get_linepos;
  E.ErrorCode := Error.Get_errorCode;
  E.FileName := Error.Get_url;
  raise E;
end;

{ TXMLDocHelper }

function TXMLDocHelper.AddAttribute(Node: IXMLDOMNode; AttrName,
  AttrValue: string): IXMLDOMAttribute;
var
  DE: IXMLDOMElement;
begin
  DE := (Node as IXMLDOMElement);
  DE.setAttribute(AttrName, AttrValue);
  Result := DE.getAttributeNode(AttrName);
end;

function TXMLDocHelper.AddAttribute(Node: IXMLDOMNode; AttrName: string;
  AttrValue: OleVariant): IXMLDOMAttribute;
var
  DE: IXMLDOMElement;
begin
  DE := (Node as IXMLDOMElement);
  DE.setAttribute(AttrName, AttrValue);
  Result := DE.getAttributeNode(AttrName);
end;

procedure TXMLDocHelper.AddCDataSection(Node: IXMLDomNode; CData: string);
var
  CdataNode: IXMLDOMCDATASection;
begin
  CdataNode := Intf.createCDATASection(CData);
  Node.appendChild(CdataNode);
end;

procedure TXMLDocHelper.AddStdDateTime(RootNode: IXMLDOMNode;
  DateTime: TDateTime; WriteZeroDates: boolean = True);
var
  _Node: IXMLDOMNode;
begin
  _Node := ForceNodes(RootNode, ['DATETIME', 'DATE'], True);
  if (WriteZeroDates) or (DateTime <> 0) then _Node.text := FormatDateTime('dd.mm.yyyy', DateTime);

  _Node := NamedParent(_Node, 'DATETIME');
  _Node := ForceNodes(_Node, ['TIME'], True);
  if (WriteZeroDates) or (DateTime <> 0) then _Node.text := FormatDateTime('hh:nn:ss', DateTime);
end;

function TXMLDocHelper.AttrExists(Node: IXMLDOMNode;
  AttrName: string): boolean;
begin
  Result := (Node as IXMLDOMElement).getAttributeNode(AttrName) <> nil;
end;

function TXMLDocHelper.ForceNodes(RootNode: IXMLDOMNode;
  Nodes2Create: array of string; UseExistsNodes: boolean = False): IXMLDOMNode;
var
  i: integer;
  N: IXMLDOMNode;
  StrArr: array of string;
begin
  if not Assigned(RootNode) then RootNode := DefaultInterface.documentElement;
  Result := RootNode;

  if not UseExistsNodes then begin
    for i:=Low(Nodes2Create) to High(Nodes2create) do begin
      Result := Result.appendChild(Intf.createElement(Nodes2Create[i]));
    end;
  end
  else begin
    N := NodeByName(Result, Nodes2Create[Low(Nodes2Create)], 1);
    if not Assigned(N) then begin
      Result := Result.appendChild(Intf.createElement(Nodes2Create[Low(Nodes2Create)]));
      SetLength(StrArr, Length(Nodes2Create) - 1);
      for i:=Low(Nodes2Create) to High(Nodes2Create) - 1 do
        StrArr[i] := Nodes2Create[i+1];
      if Length(StrArr) > 0 then
        Result := ForceNodes(Result, StrArr, False);
    end
    else begin
      SetLength(StrArr, Length(Nodes2Create) - 1);
      for i:=Low(Nodes2Create) to High(Nodes2Create) - 1 do
        StrArr[i] := Nodes2Create[i+1];
      if Length(StrArr) > 0 then
        Result := ForceNodes(N, StrArr, True)
      else Result := N;
    end;
  end;
end;

function TXMLDocHelper.GetABNI(Node: IXMLDOMNode;
  AttrName: String): Integer;
var
  DE: IXMLDOMElement;
begin
  DE := Node as IXMLDomElement;
  Result := DE.getAttribute(AttrName);
end;

function TXMLDocHelper.GetABNS(Node: IXMLDOMNode;
  AttrName: String): String;
var
  DE: IXMLDOMElement;
  Val: OleVariant;
begin
  Result := '';
  DE := Node as IXMLDomElement;
  Val := DE.getAttribute(AttrName);
  if Val <> null then Result := Val;
end;

function TXMLDocHelper.GetIntf: IXMLDOMDocument2;
begin
  Result := DefaultInterface;
end;

function TXMLDocHelper.GetParseError: IXMLDOMParseError;
begin
  Result := DefaultInterface.parseError;
end;

function TXMLDocHelper.GetRoot: IXMLDOMElement;
begin
  Result := Intf.Get_documentElement;
end;


function TXMLDocHelper.GetStdDateTime(RootNode: IXMLDOMNode; RaiseOnError:boolean = True): TDateTime;
var
  DateNode, TimeNode: IXMLDOMNode;
  TimeText, DateText: string;
begin
  Result := 0;
  if not Assigned(RootNode) then begin
    if RaiseOnError then raise Exception.Create('  GetStdDateTime(NIL)')
    else exit;
  end;
  DateNode := NodeByName(RootNode, 'DATETIME\DATE');
  if not Assigned(DateNode) then
  begin
    try
      Result := XMLDate2DateTime(RootNode.Text);
//VIST
//      if length(RootNode.Text) = 10 then
//        Result := StrToDateFmt('yyyy-mm-dd', RootNode.Text)
//      else
//        Result := StrToDateTimeFmt('yyyy-mm-dd', RootNode.Text);
      Exit;
    except
      if RaiseOnError then
        raise Exception.CreateFmt('   <%s><DATETIME><DATE>',
                                [RootNode.NodeName])
      else exit;
    end;
  end;
  DateText := DateNode.Text;
  TimeNode := NodeByName(RootNode, 'DATETIME\TIME');
  if not Assigned(TimeNode) then
  begin
    if RaiseOnError then
      raise Exception.CreateFmt('   <%s><DATETIME<TIME>',
                              [RootNode.NodeName])
    else exit;
  end;
  TimeText := TimeNode.Text;
  if (DateText = '') and (TimeText = '') then Result := 0
  else
    try
{
function XMLDate2DateTime(XMLDate: string;
  DefDateTime: TDateTime): TDateTime;
var
  y,m,d : integer;
begin
    y := StrToIntDef(copy(XMLDate,1,4),0);
    m := StrToIntDef(copy(XMLDate,6,2),0);
    d := StrToIntDef(copy(XMLDate,9,2),0);
    if (y <= 0) or (m <= 0) or (d <= 0) then
      result := DefDateTime
    else
      try
        result := EncodeDate(y,m,d);
      except
        result := DefDateTime;
      end;
end;
}
      Result := EncodeDate(StrToInt(ExtractWord(3, DateNode.Text, ['.'])), //Year
                           StrToInt(ExtractWord(2, DateNode.Text, ['.'])), //Month
                           StrToInt(ExtractWord(1, DateNode.Text, ['.']))) + //Day
                EncodeTime(StrToInt(ExtractWord(1, TimeNode.Text, [':'])), //Hour
                           StrToInt(ExtractWord(2, TimeNode.Text, [':'])), //Minute
                           StrToInt(ExtractWord(3, TimeNode.Text, [':'])), //Second
                           0); //Millisecond

    except
      if RaiseOnError then
        raise EConvertError.CreateFmt('   /.'#13''#13+
                                '<DATE>%s</DATE> <TIME>%s</TIME>',
                                [DateNode.Text,
                                 TimeNode.Text])
      else Result := 0;
    end;
end;

function TXMLDocHelper.GetValidateOnParse: boolean;
begin
  Result := DefaultInterface.Get_validateOnParse;
end;

function TXMLDocHelper.InternalSearchForNode(RootNode: IXMLDOMNode;
  NodeName: string; MaxDepth, CurDepth: Integer;
  NodeType: DOMNodeType): IXMLDOMNode;
var
  i: longint;
begin
  Result := nil;
  if not Assigned(RootNode) then begin
    RootNode := GetRoot;
  end;
  if not Assigned(RootNode) then Exit;
  if not RootNode.hasChildNodes then Exit;
  for i:=0 to RootNode.childNodes.length - 1 do begin
    if RootNode.childNodes[i].nodeType = NodeType then begin
      if RootNode.childNodes[i].nodeName = NodeName then begin
        Result := RootNode.childNodes[i];
        break;
      end;
    end;
    if (RootNode.childNodes[i].hasChildNodes) and ((CurDepth < MaxDepth) or (MaxDepth = 0)) then begin
      Result := InternalSearchForNode(RootNode.childNodes[i], NodeName, NodeType, MaxDepth, CurDepth + 1);
      if Result <> nil then break;
    end;
  end;
end;

function TXMLDocHelper.InternalSearchForNodeDepth(RootNode: IXMLDOMNode;
  NodeName: string; MaxDepth, CurDepth: Integer;
  NodeType: DOMNodeType): IXMLDOMNode;
var
  i: longint;
begin
  Result := nil;
  if not Assigned(RootNode) then begin
    RootNode := GetRoot;
  end;
  for i:=0 to RootNode.childNodes.length - 1 do begin
    if RootNode.childNodes[i].nodeType = NodeType then begin
      if AnsiSameText(RootNode.childNodes[i].nodeName, NodeName) then begin
        Result := RootNode.childNodes[i];
        break;
      end;
    end;
    if (RootNode.childNodes[i].hasChildNodes) and ((CurDepth < MaxDepth) or (MaxDepth = 0)) then begin
      Result := InternalSearchForNodeDepth(RootNode.childNodes[i], NodeName, MaxDepth, CurDepth + 1, NodeType);
      if Result <> nil then break;
    end;
  end;
end;

procedure TXMLDocHelper.LoadFromFile(Source: String; RaiseOnParseError: Boolean = False);
begin
  Intf.load(OleVariant(Source));

  if RaiseOnParseError and Assigned(Intf.parseError) and
    (Intf.parseError.errorCode <> 0) then
  begin
    raise Exception.CreateFmt(
      'failed to parse XML document [FileName: %s, Error: %s]',
      [Source, Intf.parseError.reason]);
  end;
end;

procedure TXMLDocHelper.LoadFromStream(Source: TStream);
var
  SA: TStreamAdapter;
begin
  SA := TStreamAdapter.Create(Source, soReference);
  DefaultInterface.load(SA as IStream);
  if ValidateOnParse then
    if Intf.parseError.errorCode <> 0 then XMLError(Intf.parseError);
end;

procedure TXMLDocHelper.LoadFromString(Source: string);
var
  SA: TStreamAdapter;
  Strm: TStringStream;
begin
  Strm := TStringStream.Create(Source);
  try
    SA := TStreamAdapter.Create(Strm, soReference);
    DefaultInterface.load(SA as IStream);
    if ValidateOnParse then
      if Intf.parseError.errorCode <> 0 then XMLError(Intf.parseError);
  finally
    Strm.Free;
  end;
end;

function TXMLDocHelper.NamedParent(ChildNode: IXMLDOMNode;
  RootName: string): IXMLDOMNode;
begin
  Result := ChildNode;
  while Result.ParentNode <> nil do begin
    Result := Result.ParentNode;
    if Result.NodeName = RootName then exit;
  end;
  Result := nil;
end;

function TXMLDocHelper.NodeByName(RootNode: IXMLDOMNode; NodeName: string;
                                  MaxDepth: longint = 0; includeSelf: boolean = False): IXMLDOMNode;
var
  i: integer;
begin
  if not assigned(RootNode) then
    Result := Root
  else
    Result := RootNode;

  if NodeName <> '' then
  begin
    for i := 1 to WordCount(NodeName, ['/', '\', '.']) do
    begin
      if (i = 1) and IncludeSelf and (Result.nodeName = ExtractWord(i, NodeName, ['/', '\', '.'])) then continue;
      Result := InternalSearchForNode(Result, ExtractWord(i, NodeName, ['/', '\', '.']),
                                      NODE_ELEMENT, MaxDepth, 1);
      if not Assigned(Result) then break;
    end;

    if includeSelf and not Assigned(Result) then begin
      if RootNode.nodeName = NodeName then
        Result := RootNode;
    end;
  end;
end;

function TXMLDocHelper.NodeByNameDepth(RootNode: IXMLDOMNode; NodeName: string;
  MaxDepth: longint = 0; includeSelf: boolean = False): IXMLDOMNode;
var
  i: integer;
begin
  if not assigned(RootNode) then
    Result := Root
  else
    Result := RootNode;

  if NodeName <> '' then begin
    for i := 1 to WordCount(NodeName, ['/', '\', '.']) do
    begin
      if (i = 1) and IncludeSelf and (Result.nodeName = ExtractWord(i, NodeName, ['/', '\', '.'])) then continue;
      Result := InternalSearchForNodeDepth(Result, ExtractWord(i, NodeName, ['/', '\', '.']),
                                      MaxDepth, 1, NODE_ELEMENT);
      if not Assigned(Result) then break;

    end;

    if includeSelf and not Assigned(Result) then
    begin
      if RootNode.nodeName = NodeName then Result := RootNode;
    end;
  end;
end;

procedure TXMLDocHelper.SaveToFile(FileName: string; Overwrite: boolean = True);
var
  fs: TFileStream;
  flags: longint;
begin
  if FileExists(FileName) and not Overwrite then flags := fmOpenReadWrite
                                            else Flags := fmCreate;
  fs := TFileStream.Create(FileName, Flags);
  try
    SaveToStream(fs, Overwrite);
  finally
    fs.Free;
  end;
end;

procedure TXMLDocHelper.SaveToStream(Stream: TStream; Overwrite: boolean = True);
var
  SA: IStream;
begin
  if ValidateOnSave then begin
    Intf.validate;
    if ParseError.errorCode <> 0 then XMLError(ParseError);
  end;

  SA := TStreamAdapter.Create(Stream, soReference);
  if not Overwrite then Stream.Seek(1, soFromEnd);
  Intf.save(SA);
end;

procedure TXMLDocHelper.SetABNI(Node: IXMLDOMNode; AttrName: String;
  const Value: Integer);
var
  DE: IXMLDOMElement;
begin
  DE := Node as IXMLDomElement;
  DE.setAttribute(AttrName, Value);
end;

procedure TXMLDocHelper.SetABNS(Node: IXMLDOMNode; AttrName: String;
  const Value: String);
var
  DE: IXMLDOMElement;
begin
  DE := Node as IXMLDomElement;
  DE.setAttribute(AttrName, Value);
end;

procedure TXMLDocHelper.SetValidateOnParse(const Value: boolean);
begin
  DefaultInterface.Set_validateOnParse(Value);
end;

function TXMLDocHelper.NodeExists(RootNode: IXMLDOMNode; NodeName: string): boolean;
begin
  Result := Assigned(NodeByName(RootNode, NodeName, 0));
end;

Procedure DecorateXML(XmlDocName: string);
var
  writer: IMXWriter;
  reader: ISAXXMLReader;
  ws: WideString;
begin
  writer := CreateComObject(CLASS_MXXMLWriter) as IMXWriter;
  writer.encoding := 'windows-1251';
  reader := CreateComObject(CLASS_SAXXMLReader) as ISAXXMLReader;
  reader.putContentHandler(writer as ISAXContentHandler);
  reader.putDTDHandler(writer as ISAXDTDHandler);
//  reader.putProperty('http://xml.org/sax/properties/declaration-handler', writer);
//  reader.putProperty('http://xml.org/sax/properties/lexical-handler', writer);
  writer.Output := '';
  writer.indent := True;
  ws := XMLDocName;
//  reader.parseURL(ws);
  ShowMessage(writer.output);
end;

function TXMLDocHelper.AppendSibling(Node: IXMLDOMNode;
  NodeName: string): IXMLDOMNode;
begin
  if Assigned(Node.NextSibling)
  then Result := Node.parentNode.insertBefore(Intf.createElement(NodeName), Node.nextSibling)
  else result := Node.parentNode.appendChild(Intf.createElement(NodeName));
end;

function TXMLDocHelper.GetDateTime(RootNode: IXMLDOMNode;
  NodeName: string): TDateTime;
Var
  N : IXMLDOMNode;
begin
  N := NodeByName(RootNode, NodeName, 1);
  if assigned(N) then begin
    result:= GetStdDateTime(N);
  end else begin
    result := 0;
  end;
end;

function TXMLDocHelper.GetText(RootNode: IXMLDOMNode;
       NodeName: string = '';
       RaiseIfNotFound : boolean = False): string;
Var
  N : IXMLDOMNode;
begin
  N := NodeByName(RootNode, NodeName, 1);
  if assigned(N) then begin
    result:= N.Text
  end else begin
    if RaiseIfNotFound then begin
      raise EXMLMustGetValue.Create('   XML ' + GetNodePathName( RootNode, NodeName));
    end;
    result := '';
  end;
end;

function TXMLDocHelper.MustGetText(RootNode: IXMLDOMNode;
  NodeName: string): string;
begin
  result := GetText( RootNode, NodeName, True );
end;

function TXMLDocHelper.MustGetDateTime(RootNode: IXMLDOMNode;
  NodeName: string): TDateTime;
Var
  N : IXMLDOMNode;
  s : string;
begin
  s := '';
  if assigned(RootNode) then s := RootNode.nodeName;
  N := NodeByName(RootNode, NodeName);
  if not assigned(N) then
     raise EXMLMustGetValue.Create('   XML ' + s + '/'+ NodeName);
  result:= GetStdDateTime(N);
end;

function TXMLDocHelper.MustGetDecimalNumber(RootNode: IXMLDOMNode; NodeName: string = ''): Currency;
begin
  result := GetDecimalNumber( RootNode, NodeName, True );
end;

Function InListUC(v:string; const TestVal:array of string):boolean;
Var
   k : integer;
begin
   v := UpperCase(v);
   for k:=0 to Length(TestVal)-1 do begin
      if v= UpperCase( TestVal[k] ) then begin
         result:=True;
         exit;
      end;
   end;
   result:=False;
end;

procedure TXMLDocHelper.AddNodesFromDS(RootNode: IXMLDOMNode; DS: TDataSet; FieldList: array of string;
                                       ISExcludeFieldList : boolean; UseExistsNodes: boolean);
var
  i : integer;
  N : IXMLDOMNode;

  function TrueFieldName(FieldName: string): string;
  begin
    Result := UpperCase(FieldName);
    if pos('.', Result) > 0 then
      Result := copy(Result, pos('.', Result) + 1, length(Result));
  end;

  procedure AddFieldNode(f : TField);
  begin
    N := ForceNodes(RootNode, [TrueFieldName(f.FieldName)] , UseExistsNodes);
    if f.IsNull then begin
//      if (f.DataType = ftDate) or
//         (f.DataType = ftTime) or
//         (f.DataType = ftDateTime) then begin
//        AddStdDateTime(N, f.AsDateTime, False);
//      end;
    end else begin
      if (f.DataType = ftDate) or
         (f.DataType = ftTime) or
         (f.DataType = ftDateTime) then
      begin
        N.text := FormatDateTime('yyyy-mm-dd', f.AsDateTime);
      end else begin
        N.text := f.AsString;
      end;
    end;
  end;

begin
  if ISExcludeFieldList then begin
    for i := 0 to DS.FieldCount - 1 do begin
      if not inListUC(TrueFieldName(DS.Fields[i].FieldName), FieldList) then begin
        AddFieldNode(DS.Fields[i]);
      end;
    end;
  end else begin
    for i := 0 to length(FieldList) - 1 do begin
     AddFieldNode(DS.FieldByName(FieldList[i]));
    end;
  end;
end;

procedure TXMLDocHelper.ReplDSFromNodes(RootNode: IXMLDOMNode;
  DS: TDataSet; FieldList: array of string;
  ISExcludeFieldList : boolean; UpCase: boolean);
Var
  i : integer;
  N : IXMLDOMNode;

  Procedure ReplNodeField(f : TField);
  var
    DecSep: Char;
  begin
    if UpCase then begin
      N := NodeByName(RootNode, UpperCase( f.FieldName ) );
    end else begin
      N := NodeByName(RootNode, LowerCase( f.FieldName ) );
    end;
    if assigned(N) then begin
      if N.Text = '' then
        f.Value := null
      else begin
        if (f.DataType = ftDate) or
           (f.DataType = ftTime) or
           (f.DataType = ftDateTime) then begin
//VIST
//          f.AsDateTime := StrToDateFmt('yyyy-mm-dd', N.Text);
           f.AsDateTime := XMLDate2DateTime(N.Text);
        end else if f is TNumericField then
        begin
          DecSep := DecimalSeparator;
          try
            DecimalSeparator := '.';
            f.AsString := N.Text;
          except
            DecimalSeparator := ',';
            f.AsString := N.Text;
          end;
          DecimalSeparator := DecSep;
        end else
         f.AsString := N.Text;
      end;
    end;
  end;
begin
  if ISExcludeFieldList then begin
    for i:=0 to DS.FieldCount-1 do begin
      if not inListUC( DS.Fields[i].FieldName, FieldList) then begin
        ReplNodeField( DS.Fields[i] );
      end;
    end;
  end else begin
    for i:=0 to length(FieldList) -1 do begin
     ReplNodeField( DS.FieldByName( FieldList[i] ) );
    end;
  end;
end;

Procedure CreateXMLFileByDS(const BaseElement:string; DS: TDataSet;
   FieldList:array of string;ISExcludeFieldList : boolean;
   const OutFileName:string );
var
  XML: TXMLDocHelper;
begin
  XML := TXMLDocHelper.Create(nil);
  try
    XML.PrepareBlank( BaseElement );
    XML.AddNodesFromDS( XML.Root, DS, FieldList, ISExcludeFieldList, False);
    XML.SaveToFile(OutFileName);
  finally
    XML.Free;
  end;
end;

Procedure ReplDSFromXMLFile(const XMLFileName: string; DS: TDataSet);
var
  XML: TXMLDocHelper;
begin
  XML := TXMLDocHelper.Create(nil);
  try
    XML.ValidateOnParse := False;
    XML.LoadFromFile( XMLFileName );
    XML.ReplDSFromNodes( XML.Root, DS, [], True);
  finally
    XML.Free;
  end;
end;

procedure TXMLDocHelper.PrepareBlank(BaseElement: string; UpCase: boolean = true; isDOCTYPE: boolean = True);
var
  BaseEl, str: string;
begin
  ValidateOnParse := False;
  if UpCase then begin
    BaseEl := UpperCase(BaseElement);
  end
  else begin
    BaseEl := BaseElement;
  end;

  str := '<?xml version="1.0" encoding="Windows-1251"?>'#10#13;
  if isDOCTYPE then
    str := str + '<!DOCTYPE '+ BaseEl + '>'#10#13;
  str := str + '<'+ BaseEl + '/>';

  LoadFromString(str);
end;

function TXMLDocHelper.IsNullDate(RootNode: IXMLDOMNode): boolean;
var
  DateNode, TimeNode: IXMLDOMNode;
  TimeText, DateText: string;
begin
  if not Assigned(RootNode)
  then raise Exception.Create('  IsNullDate(NIL)');
  DateNode := NodeByName(RootNode, 'DATETIME\DATE');
  if not Assigned(DateNode) then begin
    raise Exception.CreateFmt('   <%s><DATETIME<DATE> (IsNullDate)',
                              [RootNode.NodeName]);
  end;
  DateText := DateNode.Text;
  TimeNode := NodeByName(RootNode, 'DATETIME\TIME');
  if not Assigned(TimeNode) then begin
    raise Exception.CreateFmt('   <%s><DATETIME<TIME> (IsNullDate)',
                              [RootNode.NodeName]);
  end;
  TimeText := TimeNode.Text;
  Result := (DateText = '') and (TimeText = '');
end;

function TXMLDocHelper.TransformWithXSL(const XSLTemplate: string): String;
var
  XSLDoc: IXMLDomDocument2;
  strm: IStream;
  ms: TMemoryStream;
  Buffer: TBytes;
  Encoding: TEncoding;
  Node, ChildNode: IXMLDOMNode;
begin
  XSLDoc := CreateCOMObject(ServerData.ClassID) as IXMLDomDocument2;
  XSLDoc.validateOnParse := False;
  if FileExists(XSLTemplate) then
    XSLDoc.load(OleVariant(XSLTemplate))
  else
    XSLDoc.loadXML(XSLTemplate);
  //  Result := Intf.transformNode(XSLDoc.documentElement);
  //  html  ,    xsl    utf-16
  //         ,    
  ms := TMemoryStream.Create;
  strm := TStreamAdapter.Create(ms, soOwned) as IStream;
  Intf.transformNodeToObject(XSLDoc.documentElement, strm);

  ms.Position := 0;
  SetLength(Buffer, ms.Size);
  ms.ReadBuffer(Pointer(Buffer)^, Length(Buffer));
  Encoding := nil;

//******************       .   ,    XSL. Smirnov. ******************
  Node := XSLDoc.selectSingleNode('xsl:stylesheet');
  ChildNode := Node.selectSingleNode('xsl:output');
  if ChildNode <> nil then begin
    ChildNode := ChildNode.attributes.getNamedItem('encoding');
    if ChildNode <> nil then 
      Encoding := TEncoding.GetEncoding(LowerCase(ChildNode.text));
  end
  else begin
  {$IFDEF CORPDB}
     Encoding := TEncoding.GetEncoding('utf-16'); //       ..     . . ******************
  {$ENDIF}
  end;
   
  if Assigned(Encoding) then begin
    try
      Result := Encoding.GetString(Buffer);
    finally
      FreeAndNil(Encoding);
    end
  end else
    raise Exception.Create('TransformWithXSL:  XSL    (   "encoding")');
end;

function TXMLDocHelper.ValidateAgainstExternalXSD(XSDFileName: string): IXMLDOMParseError;
var
  HaveXSIAttr, HaveNNAttr: boolean;
  IXML: IXMLDOMDocument2;
  strm: TMemoryStream;
  SA: IStream;
begin
  if not AnsiSameText(copy(XSDFileName, 1, 7), 'file://') then begin
    if not AnsiSameText(XSDFileName, ExtractFileName(XSDFileName)) then begin
      XSDFileName := 'file://' + StringReplace(XSDFileName, '\', '//', [rfReplaceAll]);
    end;
  end;

  if Root <> nil then begin
    HaveXSIAttr := AttrExists(Root, XMLNS_XSI_ATTR_NAME);
    if not HaveXSIAttr then begin
      ABNS[Root, XMLNS_XSI_ATTR_NAME] := XMLNS_XSI_ATTR_VALUE;
    end;
    HaveNNAttr := AttrExists(Root, XSI_NONAMESPACE_ATTR_NAME);
    ABNS[Root, XSI_NONAMESPACE_ATTR_NAME] := Format(XSI_NONAMESPACE_ATTR_VALUE, [XSDFileName]);
  end;

  strm := TMemoryStream.Create;
  SA := TStreamAdapter.Create(strm, soOwned);
  Intf.save(SA);
  strm.Seek(0, soFromBeginning);

  IXML := CreateMSXML2DOMDocument as IXMLDOMDocument2;
  IXML.async := False;
  IXML.resolveExternals := True;
  IXML.ValidateOnParse := true;
  IXML.load(SA);
  Result := IXML.ParseError;
  if not HaveXSIAttr then begin
    DeleteAttribute(Root, XMLNS_XSI_ATTR_NAME);
  end;
  if not HaveNNAttr then begin
    DeleteAttribute(Root, XSI_NONAMESPACE_ATTR_NAME);
  end;
end;

procedure TXMLDocHelper.DeleteAttribute(Node: IXMLDOMNode; AttrName: string);
begin
  (Node as IXMLDOMElement).removeAttribute(AttrName);
end;

function TXMLDocHelper.GetAsString: string;
var
  st: TStringStream;
begin
  st := TStringStream.Create('');
  SaveToStream(st);
  Result := st.DataString;
  st.Free;
end;

function TXMLDocHelper.AppendChild(Root: IXMLDomNode;
  ChildNodeName: string): IXMLDomNode;
begin
  Result := Intf.createElement(ChildNodeName);
  Root.appendChild(Result);
end;

function TXMLDocHelper.GetISODate(Node: IXMLDomNode): TDateTime;
var
  s: string;
begin
  s := Node.text;
  if Length(s) <> 10 then begin
    raise EParserError.CreateFmt('     <>   <%s>%s</%s>',
      [String(Node.nodeName), s, String(Node.nodeName)]);
  end;
  try
    Result := EncodeDate(StrToInt(Copy(s, 1, 4)), StrToInt(Copy(s, 6, 2)), StrToInt(Copy(s, 9, 2)));
  except
    raise EParserError.CreateFmt('     <>   <%s>%s</%s>',
      [String(Node.nodeName), s, String(Node.nodeName)]);
  end;
end;

procedure TXMLDocHelper.SetISODate(Node: IXMLDomNode; Date: TDateTime);
begin
  Node.Text := FormatDateTime('yyyy-mm-dd', Date);
end;

function TXMLDocHelper.GetISODateTime(Node: IXMLDomNode): TDateTime;
var
  s: string;
begin
  s := Node.text;
  if Length(s) <> 19 then begin
    raise EParserError.CreateFmt('     <  >   <%s>%s</%s>',
      [String(Node.nodeName), s, String(Node.nodeName)]);
  end;
  try
    Result := EncodeDateTime(StrToInt(Copy(s, 1, 4)), StrToInt(Copy(s, 6, 2)), StrToInt(Copy(s, 9, 2)),
      StrToInt(Copy(s, 12, 2)), StrToInt(Copy(s, 15, 2)), StrToInt(Copy(s, 18, 2)), 0);
  except
    raise EParserError.CreateFmt('     <  >   <%s>%s</%s>',
      [String(Node.nodeName), s, String(Node.nodeName)]);
  end;
end;

function TXMLDocHelper.GetISODateOrDateTime(Node: IXMLDomNode; NodeName: string = '';
  RaiseIfNotFound: boolean = False): TDateTime;
Var
  N: IXMLDomNode;
  function _CheckAssignedN():boolean;
  begin
    result := assigned(N);
    if not assigned(N) then begin
      if RaiseIfNotFound then begin
        raise EXMLMustGetValue.Create('   XML ' + GetNodePathName( Node, NodeName));
      end;
    end;
  end;
begin
  N := NodeByName( Node, NodeName );
  Result := 0;
  if not _CheckAssignedN() then exit;
  if NodeExists( N, 'Dt') then begin
    result := GetISODate( NodeByName( N, 'Dt' ) );
  end
  else begin
    N := NodeByName( N, 'DtTm');
    if not _CheckAssignedN() then exit;
    result := GetISODateTime( N );
  end;
end;

function TXMLDocHelper.GetPartadDate(Node: IXMLDomNode; NodeName: string = '';
          RaiseIfNotFound: boolean = False): TDateTime;
Var
  N: IXMLDomNode;
  function _CheckAssignedN: boolean;
  begin
    Result := Assigned(N);
    if not Assigned(N) then begin
      if RaiseIfNotFound then begin
        raise EXMLMustGetValue.Create('   XML ' + GetNodePathName(Node, NodeName));
      end;
    end;
  end;
begin
  N := NodeByName(Node, NodeName);
  Result := 0;
  if not _CheckAssignedN() then exit;
  Result := GetISODate(N);
end;


procedure TXMLDocHelper.SetISODateTime(Node: IXMLDomNode; Date: TDateTime);
begin
  Node.Text := FormatDateTime('yyyy-mm-dd hh:nn:ss', Date);
end;

procedure TXMLDocHelper.SetDecimalNumber(Node: IXMLDomNode; Value: Currency);
begin
  Node.text := CurrToStrF(Value, ffFixed, FDecimalSettings.CurrencyDecimals, FDecimalSettings);
end;

function TXMLDocHelper.GetDecimalNumber(RootNode: IXMLDomNode;
                  NodeName: string = '';
                  RaiseIfNotFound : boolean = False): Currency;
Var
  s : string;
begin
  s := GetText(RootNode, NodeName, RaiseIfNotFound);
  if s = '' then begin
    Result := 0;
    exit;
  end;
  try
    Result := StrToCurr(s, FDecimalSettings);
  except
    on E: Exception do begin
      E.Message := Format('     %s "%s"  ',
        [GetNodePathName(RootNode, NodeName) , s]);
      raise;
    end;
  end;
end;

function TXMLDocHelper.GetEncoding: string;
var
  PrologStr: string;
  ResultStr: string;
  EncPos: integer;
begin
  if not Assigned(DefaultInterface.childNodes) then begin
    Result := '';
  end
  else begin
    if DefaultInterface.childNodes.length = 0 then begin
      Result := '';
    end
    else begin
      PrologStr := DefaultInterface.childNodes.item[0].text;
      EncPos := pos('encoding', PrologStr);
      if EncPos > 0 then begin
        ResultStr := Trim(Copy(PrologStr, EncPos + length('encoding'), length(PrologStr)));
        Delete(ResultStr, 1, 1);
        Delete(ResultStr, 1, Pos('"', ResultStr));
        ResultStr := Copy(ResultStr, 1, Pos('"', ResultStr) - 1);
        Result := ResultStr;
      end
      else begin
        Result := '';
      end;
    end;
  end;
end;

constructor TXMLDocHelper.Create(AOwner: TComponent);
begin
  inherited;
  FDecimalSettings.DecimalSeparator := '.';
  FDecimalSettings.ThousandSeparator := #0;
  FDecimalSettings.CurrencyDecimals := 2;
  FDecimalSettings.CurrencyString := '';
  FDecimalSettings.CurrencyFormat := 0;
  FDecimalSettings.NegCurrFormat := 9;
end;

function TXMLDocHelper.GetNodePathName(RootNode: IXMLDomNode;
  NodeName: string): string;
begin
  result := '';
  if assigned(RootNode) then result := RootNode.nodeName;
  result := result + IfThen(NodeName = '', '', '/' + NodeName);
end;


function CompareXMLFile( XMLFile1, XMLFile2: string;
    KeyElements:array of string;
    ExcludeElement:array of variant;
    File1Name : string = '';
    File2Name : string = '' ): string;
var
  XML1, XML2: TXMLDocHelper;
  ErrList : TStringList;
  CurrKey : array of string;
  s : string;
  AFile1, AFile2 : string;

  function _GoNextField(N : IXMLDOMNode): IXMLDOMNode;
  begin
    N := N.nextSibling;
    while (N <> nil) and inlist(N.nodeName, ExcludeElement) do N := N.NextSibling;
  end;

  function _AddError(const s:string):boolean;
  Var
    k : integer;
  begin
    result := True;
    ErrList.Add(s);
    for k := 0 to length(CurrKey)-1 do begin
      if CurrKey[k] <> '' then begin
        ErrList.Add(' '+KeyElements[k]+'  '+CurrKey[k]);
      end;
    end;
    ErrList.Add('');
    if ErrList.Count > 40 then begin
      result := False;
      ErrList.Add('   -    ');
    end;
  end;

  Function _CompareChild(const N1, N2 : IXMLDOMNode; const ElementPath : string) : boolean;
  Var
    k1, k2, j, Max1, Max2 : integer;
    _N1, _N2 : IXMLDOMNode;
    newPath, s1, s2 : string;
  begin
    result := False;

    k1 := -1;
    k2 := -1;
    Max1 := N1.childNodes.length - 1;
    Max2 := N2.childNodes.length - 1;
    while True do begin
      while True do begin //  
        k1 := k1 + 1;
        if (k1 > Max1) then break;
        _N1 := N1.childNodes[k1];
        s := _N1.nodeName;
        newPath := ElementPath + '\' + s;
        if not inlist( NewPath, ExcludeElement ) then break;
      end;
      while True do begin //  
        k2 := k2 + 1;
        if (k2 > Max2) then break;
        _N2 := N2.childNodes[k2];
        s := _N2.nodeName;
        newPath := ElementPath + '\' + s;
        if not inlist( NewPath, ExcludeElement ) then break;
      end;
      if (k1 > Max1) and (k2 > Max2) then break;
      if (k1 > Max1) or (k2 > Max2) then begin
        _AddError('      '+ ElementPath);
        exit;
      end;

      for j:=0 to length(KeyElements)-1 do begin
        if newPath = KeyElements[j] then begin
          CurrKey[j] := _N1.text;
        end;
//        if newPath <> Left(KeyElements[j], length(s)) then begin
//          CurrKey[j] := '';
//        end;
      end;

      s := _N1.nodeName;
      if s <> _N2.nodeName then begin
        if not _AddError('    '+s + AFile1 +
                       '  '+_N2.nodeName+ AFile2 +
                       '   '+ ElementPath) then exit;
        continue;
      end;

      s1 := VarToStr(_N1.nodeValue);
      s2 := VarToStr(_N2.nodeValue);
      if s1 <> s2 then begin
        try
          if StrToCurr( s1 ) <> StrToCurr( s2 ) then Abort;
        except
          if not _AddError('    '+ ElementPath +#10#13+
              ' : '+ s1 + AFile1 + '  '+
                                      s2 + AFile2 ) then exit;
          continue;
        end;
      end;

      if not _CompareChild( _N1, _N2, newPath) then exit; //   
    end;
    result := True;
  end;

begin
  AFile1 := '';
  AFile2 := '';
  if (File1Name <> '') and (File2Name <> '') then begin
    AFile1 := ' ('+File1Name+') ';
    AFile2 := ' ('+File2Name+') ';
  end;
  XML1 := TXMLDocHelper.Create(nil);
  XML2 := TXMLDocHelper.Create(nil);
  ErrList := TStringList.Create;
  try
    XML1.ValidateOnParse := False;
    XML1.LoadFromFile( XMLFile1 );
    XML2.ValidateOnParse := False;
    XML2.LoadFromFile( XMLFile2 );
    setlength( CurrKey, length( KeyElements ));

    if (XML1.Root = nil) or (XML2.Root = nil) then begin
      _AddError('     XML');
      exit;
    end;
    _CompareChild(XML1.Root, XML2.Root, XML1.Root.nodeName );
  finally
    XML1.Free;
    XML2.Free;
    result := ErrList.Text;
    ErrList.Free;
  end;
end; //CompareXML..

threadvar
  InBodyOfScan: boolean;

procedure PreScanXML;
begin
  InBodyOfScan := False;
end;

Function  ScanXML( var CurrentNode:IXMLDOMNode; ScanNodeName : string) : boolean;
begin
  if InBodyOfScan then begin
     CurrentNode := CurrentNode.nextSibling;
  end
  else begin
    InBodyOfScan := True;
  end;
  Result := Assigned(CurrentNode) and (CurrentNode.nodeName = ScanNodeName);
end;

procedure TXMLDocHelper.RemoveEmptyNodes();
{  function AttributesIsEmpty(Node: IXMLDomNode): boolean;
  var
    i: integer;
  begin
    Result := Node.attributes.length = 0;
    if not Result then begin
      Result := True;
      for i := 0 to Node.attributes.length - 1 do
        if Node.attributes[i].text <> '' then begin
          Result := False;
          Break;
        end;
    end;
  end;}
//var
//  I: Longint;
//  Node: IXMLDomNode;
//  ChNodes: IXMLDOMNodeList;
//  XSLDoc: IXMLDomDocument2;
const
{  CLEARING_XSL = '<?xml version="1.0" encoding="Windows-1251"?>'+
                 '<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">'+
                 '<xsl:output method="xml"/>'+
                 '<xsl:template match="*">'+
                    '<xsl:if test="count(@*) &gt; 0 or count(node()) &gt; 0">'+
                      '<xsl:copy>'+
                        '<xsl:apply-templates select="@* | node()" />'+
                      '</xsl:copy>'+
                    '</xsl:if>'+
                 '</xsl:template>'+
                  '<xsl:template match="@* | text() | processing-instruction()">'+
                    '<xsl:copy />'+
                 '</xsl:template>'+
                 '</xsl:stylesheet>';}

  CLEARING_XSL = '<?xml version="1.0" encoding="windows-1251"?>'+
                 '<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">'+
                   '<xsl:output method="xml" indent="yes" encoding="Windows-1251"/>'+
                   '<xsl:strip-space elements="*"/>'+
                   '<xsl:template match="node()[descendant-or-self::text()]|@*">'+
                      '<xsl:copy>'+
                        '<xsl:apply-templates select="node()|@*"/>'+
                      '</xsl:copy>'+
                   '</xsl:template>'+
                 '</xsl:stylesheet>';

begin
  RemoveEmptyNodesQuick;
  Exit;

  TransformSelfWithXSL(CLEARING_XSL);

{  ChNodes := RootNode.childNodes;
  for I := 0 to ChNodes.length - 1 do begin
    Node := ChNodes.nextNode;

    if (Assigned(Node))
      and (Node.text = '')
      and (((not Node.hasChildNodes) and AttributesIsEmpty(Node)) //  children   (   )
        or ((Node.childNodes.length = 1) and (Node.firstChild.nodeType = NODE_TEXT))) //  ,   child
    then
      RootNode.removeChild(Node)
    else
      RemoveEmptyNodes(Node);
  end;

  if (RootNode.text = '') and ((not RootNode.hasChildNodes) and AttributesIsEmpty(RootNode)) and assigned(RootNode.parentNode) then
    RootNode.parentNode.removeChild(RootNode);
}
end;

function GetRootNodeName(XMLFileName: string): string;
begin                                                                                    
  Result := '';
  if FileExists(XMLFileName) then
    with TXMLDocHelper.Create(nil) do
    try
      ValidateOnParse := False;
      LoadFromFile(XMLFileName);
      if Assigned(Root) then Result := Root.tagName else Result := '';
    finally
      Free;
    end;
end;

procedure GetXMLRootVersion(XMLFileName: string; var RootName, Version: string);
begin
  if FileExists(XMLFileName) then begin
    with TXMLDocHelper.Create(nil) do
    try
      ValidateOnParse := False;
      LoadFromFile(XMLFileName);
      if not Assigned(Root) then
        raise Exception.Create('     xml-');
      RootName := Root.tagName;

      if not Assigned(NodeByName(Root, 'version', 1)) then
        raise Exception.Create('    xml-');
      Version := NodeByName(Root, 'version', 1).Text;
    finally
      Free;
    end;
  end;  
end;

function TXMLDocHelper.ValidateAgainstDTD(DTDFileName: string): IXMLDOMParseError;
var
  sl: TStringList;
  s: WideString;
  i: integer;
  found: boolean;
begin
  s := AsString;
  sl := TStringList.Create;
  found := False;
  try
    sl.Text := AsString;
    for i:=0 to sl.Count - 1 do begin
      if Pos('<!DOCTYPE', trim(sl[i])) = 1 then begin
        sl[i] := '<!DOCTYPE ' + ExtractWord(2, SL[i], [' ', '>']) + ' SYSTEM "'+
          DTDFileName + '">' + Copy(sl[i], Pos('>', sl[i]) + 1, length(sl[i]));
        found := True;
        break;
      end;
    end;
    ValidateOnParse := False;
    if Found then begin
      LoadFromString(sl.Text);
      Result := validate;
    end
    else Result := nil;
    LoadFromString(s);
  finally
    sl.Free;
  end;
{      SL[1] := '<!DOCTYPE ' + ExtractWord(2, SL[1], [' ', '>']) + ' SYSTEM "' +
        FLoadedTEDICSamples.Values[TransFile.IdentCode] + '">';
      XML.ValidateOnParse := False;
      XML.LoadFromString(SL.Text);
      Err := XML.Intf.validate;
}
end;

procedure TXMLDocHelper.AppendXML(RootNode: IXMLDOMNode; XMLText: string);
var
  NewNode: IXMLDOMNode;
  TmpDoc: TXMLDocHelper;
begin
  // ,        root ( root   )
  // XXX:   2   parentNode,  ,   
  Assert(RootNode.parentNode.parentNode <> nil, 'this code will not work on root node...');

  //   XML      , 
  //   ,       XML 
  TmpDoc := TXMLDocHelper.Create(nil);
  try
    TmpDoc.LoadFromString(Format('<?xml version="1.0" encoding="windows-1251"?>' +
      '<%s%s>%s</%s>',
      [RootNode.nodeName,
       //     namespace     namespace
       //  ,        xmlns="" (    )
       IfThen(RootNode.ownerDocument.documentElement.namespaceURI <> '',
         ' xmlns="' + RootNode.ownerDocument.documentElement.namespaceURI + '"'),
       XMLText, RootNode.nodeName]));
    //    
    NewNode := RootNode.parentNode.insertBefore(TmpDoc.Root, RootNode);
    //   
    if Assigned(NewNode) then
      RootNode.parentNode.removeChild(RootNode);
  finally
    TmpDoc.Free;
  end;
end;

procedure TXMLDocHelper.AppendXML(Path, XMLText: string);
var
  Node: IXMLDOMNode;
begin
  Node := Self.NodeByName(Self.GetRoot, Path);
  if Assigned(Node) then
    AppendXML(Node, XMLText)
  else
    raise Exception.Create('  : ' + Path);
end;

(*
procedure _AmountMoneyForGPBInWriting(XML: TXMLDocHelper; NodeName, ccy_code:  string);

  function GetMoneyStr(Price: TBCD; name1, name2, WordAnd: string): string;
  var
    WholePart, FractionPart: string;
  begin
    WholePart := BcdToStr(Price);
    if Pos(DecimalSeparator, WholePart) > 0 then begin
      FractionPart := Copy(WholePart, Pos(DecimalSeparator, WholePart), Length(WholePart));
      Delete(WholePart, Pos(DecimalSeparator, WholePart), Length(WholePart));
    end
    else begin
      FractionPart := '';
    end;

    Result := Trim(NumToString(StrToBcd(WholePart), 1)) + name1 + WordAnd +
              AddChar('0', Copy(FractionPart, 1, 2), 2) + name2;

  end;

  function GetAmountForCCY(ccy_code: string ; aNode : IXMLDOMNode) :string;
  var
    PriceBCD: TBCD;
  begin
    Result := '';
    PriceBCD := XML.GetBCDNumber(aNode);

    if  (ccy_code='RUB') or (ccy_code='RUR') then begin
      Result := GetMoneyStr(PriceBCD,' . ',' .', '');
    end else if ccy_code='USD' then begin
      Result := GetMoneyStr(PriceBCD,' .  ',' .', '');
    end else if  ccy_code='EUR' then begin
      Result := GetMoneyStr(PriceBCD,' . ',' .', '');
    end else begin
      Result := GetMoneyStr(PriceBCD, ' ' + ccy_code, '/100', '  ');
//      Result := Format('%s %s  %s/100', [NumToString(PriceBCD, 1),
//                                         ccy_code,
//                                         AddChar('0', FloatToStrF(Frac(BcdToDouble(PriceBCD)) * 100, ffGeneral, 2, 0),2)]);
    end;
    if Result <> '' then
      Result := AnsiUpperCase(StrUtils.LeftStr(result, 1)) + StrUtils.RightStr(result, Length(result) - 1);
  end;


  procedure FindNode(RootNode: IXMLDOMNode; NodePath, ccy_code: string);
  var
    firstStr, lastStr: string;
    Node : IXMLDOMNode;
    ammountStr: string;
  begin
    firstStr := ExtractWord(1, NodePath, ['/', '\', '.']);
    if firstStr = NodePath then
      lastStr := ''
    else
      lastStr := CopyRight(NodePath, Length(firstStr) + 2);

    Node := XML.NodeByName(RootNode, firstStr, 1);
    while Assigned(Node) do begin
      if Node.nodeName = firstStr then begin
        if lastStr = '' then begin
          ammountStr := GetAmountForCCY(ccy_code,Node);
         (Node as IXMLDOMElement).setAttribute('amount_in_writing', ammountStr);
        end else
          FindNode(Node, lastStr,ccy_code);
      end;
      Node := Node.nextSibling;
    end;
  end;
begin
  FindNode(XML.Root, NodeName,ccy_code);
end;


procedure WriteMoneyForGPB(XML: TXMLDocHelper);
var
  curr_code: string;
begin
  // / /  -02 -04   
  if  (XML.Root.tagName = 'INSTRUCTION_TO_RECEIVE_DEPO') or (XML.Root.tagName = 'INSTRUCTION_TO_DELIVER_DEPO') then begin
    curr_code := XML.GetText(XML.Root,'transaction/settlement_amount/ccy_code');
    _AmountMoneyForGPBInWriting(XML,'transaction/settlement_amount/amount', curr_code);
  end;
end;

procedure AmountInWriting(FileNameIn, FileNameOut: string; NodeNames: array of string);
var
  XML: TXMLDocHelper;
  i: integer;

  procedure FindNodes(RootNode: IXMLDOMNode; NodePath: string);
  var
    firstStr, lastStr: string;
    Node : IXMLDOMNode;
    ammountStr: string;
  begin
    firstStr := ExtractWord(1, NodePath, ['/', '\', '.']);
    if firstStr = NodePath then
      lastStr := ''
    else
      lastStr := CopyRight(NodePath, Length(firstStr) + 2);

    Node := XML.NodeByName(RootNode, firstStr, 1);
    while Assigned(Node) do begin
      if Node.nodeName = firstStr then begin
        if lastStr = '' then begin
          ammountStr := NumToString(XML.GetBCDNumber(Node), 3);
          ammountStr := AnsiUpperCase(StrUtils.LeftStr(ammountStr, 1)) + StrUtils.RightStr(ammountStr, Length(ammountStr) - 1);
          (Node as IXMLDOMElement).setAttribute('amount_in_writing', ammountStr);
        end else
          FindNodes(Node, lastStr);
      end;
      Node := Node.nextSibling;
    end;
  end;
begin
  XML := TXMLDocHelper.Create(nil);
  try
    XML.ValidateOnParse := False;
    XML.LoadFromFile(FileNameIn);

    for i := 0 to High(NodeNames) do
      FindNodes(XML.Root, NodeNames[i]);

    WriteMoneyForGPB(XML);

    XML.SaveToFile(FileNameOut);

  finally
    XML.Free;
  end;
end;
*)
procedure TXMLDocHelper.SetISODateTimeEx(Node: IXMLDomNode;
  Date: TDateTime);
begin
  Node.Text :=  DateTimeToISOXML(Date);
end;

function TXMLDocHelper.GetBCDNumber(RootNode: IXMLDomNode;
  NodeName: string; RaiseIfNotFound: boolean): TBCD;
var
  s : string;
begin
  s := GetText(RootNode, NodeName, RaiseIfNotFound);
  if s = '' then begin
    Result.Precision := 0;
    exit;
  end;
  try
    Result := StrToBCD(ReplaceStr(s, '.', DecimalSeparator));
  except
    on E: Exception do begin
      E.Message := Format('     %s "%s"  ',
                          [GetNodePathName(RootNode, NodeName) , s]);
      raise;
    end;
  end;
end;

function TXMLDocHelper.GetPartadDateOrDateTime(RootNode: IXMLDomNode;
  NodeName: string; RaiseIfNotFound: boolean): TDateTime;
Var
  N: IXMLDomNode;
begin
  N := NodeByName(RootNode, NodeName);
  if not Assigned(N) then begin
    if RaiseIfNotFound then begin
      raise EXMLMustGetValue.Create('   XML ' + GetNodePathName(RootNode, NodeName));
    end
    else begin
      Result := 0;
    end;
  end
  else begin
    if NodeExists(N, 'date') then Result := GetISODate(NodeByName(N, 'date'))
    else if NodeExists(N, 'datetime') then Result := GetISODateTime(NodeByName(N, 'datetime'))
    else if RaiseIfNotFound then begin
      raise EXMLMustGetValue.Create(' XML ' + GetNodePathName(RootNode, NodeName) + '    date  datetime'#13+
        ' GetPartadDateOrDateTime');
    end
    else begin
      Result := 0;
    end;
  end;
end;

procedure TXMLDocHelper.TransformWithXSL(const XSLTemplate, FileName: string);
var
  FileStream: TFileStream;
begin
  FileStream := TFileStream.Create(FileName, fmCreate);
  try
    TransformWithXSL(XSLTemplate, FileStream);
  finally
    FileStream.Free;
  end;
end;

procedure TXMLDocHelper.TransformWithXSL(const XSLTemplate: string; Stream: TStream);
var
  XSLDoc: IXMLDomDocument2;
  StreamAdapterKeeper: IStream;
begin
  XSLDoc := CreateCOMObject(ServerData.ClassID) as IXMLDomDocument2;
  XSLDoc.validateOnParse := False;
  if FileExists(XSLTemplate) then
  begin
    XSLDoc.load(OleVariant(XSLTemplate))
  end else
  begin
    XSLDoc.loadXML(XSLTemplate);
  end;
  
  StreamAdapterKeeper := TStreamAdapter.Create(Stream, soReference);
  Intf.transformNodeToObject(XSLDoc.documentElement, StreamAdapterKeeper);
end;

function TXMLDocHelper.IsWellFormedXmlText(AnXmlText: string): boolean;
var
  xml: TXmlDocHelper;
  str: string;
begin
// MS Parser ,    ,   xml    >,    
//     &gt;    
  str := '.' + AnXmlText + '.';
  if WordCount(str, ['<']) <> WordCount(str, ['>']) then begin
    Result := False;
  end else begin
    xml := TXmlDocHelper.Create(nil);
    try
      xml.ValidateOnParse := False;
      xml.LoadFromString('<?xml version="1.0" encoding="Windows-1251"?> <_NdCSpEcIaLNoDe>' + AnXmlText + '</_NdCSpEcIaLNoDe>');
      Result := xml.Root <> nil;
    finally
      xml.Free;
    end;
  end;
end;

function TXMLDocHelper.LoadWithoutErrors: boolean;
begin
  Result := True;
  if (Assigned(ParseError) and (ParseError.errorCode <> 0)) or (not Assigned(Root)) then
    Result := False;
end;

procedure TransformXmlFile(XmlFileName, XslFileName, OutFileName: string);
var
  xmlDoc: TXMLDocHelper;
  transformedStr: AnsiString;
begin
  xmlDoc := TXMLDocHelper.Create(nil);
  try
    xmlDoc.ValidateOnParse := False;
    xmlDoc.LoadFromFile(xmlFileName);
    if Assigned(xmlDoc.ParseError) and (xmlDoc.ParseError.errorCode <> 0) then begin
      raise Exception.CreateFmt('   %s.', [xmlFileName]);
    end else begin
      transformedStr := xmlDoc.TransformWithXSL(xslFileName);
      transformedStr := StringReplace(transformedStr, 'encoding="UTF-16"', 'encoding="windows-1251"', [rfIgnoreCase]);

      try
        with TFileStream.Create(outFileName, fmCreate) do begin
          Write(transformedStr[1], length(transformedStr));
          Free;
        end;
      except
        on E: Exception do begin
          E.Message := Format('    %s'#13'%s', [outFileName, E.Message]);
          raise;
        end;
      end;
    end;
  finally
    xmlDoc.Free;
  end;
end;

procedure TXMLDocHelper.TransformSelfWithXSL(const XSLTemplate: string);
var
  XSLDoc: IXMLDomDocument2;
begin
  XSLDoc := CreateCOMObject(ServerData.ClassID) as IXMLDomDocument2;
  XSLDoc.validateOnParse := False;
  if FileExists(XSLTemplate) then
    XSLDoc.load(OleVariant(XSLTemplate))
  else
    XSLDoc.loadXML(XSLTemplate);

  Intf.transformNodeToObject(XSLDoc.documentElement, Intf);
  XSLDoc := nil;
end;

type
  TNodeInfo = Record
    index : integer;
    IsEmpty : boolean;
  end;

Type TNodeEmpterState = (Unknown, InNodeNameOpen, InNodeNameClose, InNodeData, InCDATA);

procedure _DoEmptyNodes(p: pointer; Size: integer);
Var
  NodeStack : array of TNodeInfo;
  NodeLevel : integer;
  CH : AnsiChar;
  k : integer;
  State : TNodeEmpterState;
  InComment : boolean;

procedure _CloseNode(CurIdx : integer);
Var
  m : integer;
begin
  if NodeStack[NodeLevel].IsEmpty then begin
    m := NodeStack[NodeLevel].Index;
    FillChar(PByte(p)[m], CurIdx - m + 1, #32); //  
  end
  else begin
    for m := 0 to NodeLevel-1 do begin
      NodeStack[m].IsEmpty := False;
    end;
  end;
  Dec(NodeLevel);
  if NodeLevel < -1 then raise Exception.Create('     ');
end;

begin
  NodeLevel := -1;
  State := Unknown;
  InComment := False;
  setLength(NodeStack, 0);
  k := 0;
  while k < Size - 2 do begin //-2  
    if InComment then begin
      if AnsiChar(PByte(p)[k]) + AnsiChar(PByte(p)[k + 1]) + AnsiChar(PByte(p)[k + 2]) = '-->' then begin
        InComment := False;
        inc(k, 3);
      end
      else begin
        inc(k);
      end;
      continue;
    end;

    if State = InCDATA then begin
      if AnsiChar(PByte(p)[k]) + AnsiChar(PByte(p)[k + 1]) + AnsiChar(PByte(p)[k + 1]) = ']]>' then begin
        State := Unknown;
        inc(k, 3);
      end
      else begin
        inc(k);
      end;
      continue;
    end;

    CH := AnsiChar(PByte(p)[k]);

    case CH of
      '<': begin
        if AnsiChar(PByte(p)[k + 1]) = '/' then begin //  </
          inc(k, 2);
          State := InNodeNameClose;
          continue;
        end;
        if AnsiChar(PByte(p)[k + 1]) + AnsiChar(PByte(p)[k + 2]) + AnsiChar(Pbyte(p)[k + 3]) = '!--' then begin //  <!--
          InComment := True;
          inc(k, 3);
          continue;
        end;
        if AnsiChar(PByte(p)[k + 1]) + AnsiChar(PByte(p)[k + 2]) + AnsiChar(Pbyte(p)[k + 3]) = '![C' then begin //   CDATA
          if k < Size - 8 then begin
            if AnsiChar(PByte(p)[k + 1])+AnsiChar(PByte(p)[k + 2])+AnsiChar(PByte(p)[k + 3])+AnsiChar(PByte(p)[k + 4])+
               AnsiChar(PByte(p)[k + 5])+AnsiChar(PByte(p)[k + 6])+AnsiChar(PByte(p)[k + 7])+AnsiChar(PByte(p)[k + 8]) =
               '![CDATA[' then begin
              inc(k, 8);
              State := InCDATA;
              NodeStack[NodeLevel].IsEmpty := False; //CDATA   . Baa
              continue;
            end;
          end;
        end;
        inc(NodeLevel);
        if NodeLevel + 1> length(NodeStack) then begin
          SetLength(NodeStack, NodeLevel + 1);
        end;
        With NodeStack[NodeLevel] do begin
          index := k;
          IsEmpty := True;
        end;
        State := InNodeNameOpen;
      end;
      '>':begin
        if State = InNodeNameClose then begin
          _CloseNode(k);
        end;
        State := InNodeData;
      end;
      '/': begin
        if AnsiChar(PByte(p)[k + 1]) = '>' then begin //   />
          inc(k, 1);
          _CloseNode(k); //k     
          inc(k, 1);
          State := InNodeData;
          continue;
        end;
      end;
    else
      if (State = InNodeData) and NodeStack[NodeLevel].IsEmpty then begin

        if not CharInSet(ch, [#0, #10, #13, #32, #9]) then begin
          NodeStack[NodeLevel].IsEmpty := False;
        end;
      end;
    end; //case;

    inc(k);
  end;
end;

procedure RemoveEmptyNodeQuick(const FileName, OutFileName:string);
var
  ms : TMemoryStream;
begin
  ms := TMemoryStream.Create;
  try
    ms.LoadFromFile(FileName);
    _DoEmptyNodes(ms.Memory, ms.Size );
    ms.SaveToFile(OutFileName);
  finally
    ms.Free;
  end;
end;


procedure TXMLDocHelper.RemoveEmptyNodesQuick;
Var
  s : AnsiString;
begin
  s := AnsiString(AsString);
  if s<>'' then begin
    _DoEmptyNodes(@s[1], length(s));
    LoadFromString(string(s));
  end;
end;

function TXMLDocHelper.GetNodeXPath(ANode: IXMLDomNode): string;
begin
  Result := '';

  while (ANode <> nil) and (ANode.nodeType <> NODE_DOCUMENT) do begin
    if Result <> '' then
      Result := '/' + Result;

    Result :=  ANode.nodeName + Result;

    ANode :=  ANode.parentNode;
  end;
end;

function TXMLDocHelper.GetNextNode(ANode: IXMLDomNode): IXMLDomNode;
begin
  if ANode.hasChildNodes then begin
    Result := ANode.childNodes[0];
  end
  else begin
    Result := ANode.nextSibling;

    if Result = nil then begin
      while True do begin
        ANode := ANode.parentNode;
        if ANode.nodeType = NODE_DOCUMENT then
          Exit;

        Result := ANode.nextSibling;
        if Result <> nil then
          Exit;
      end;
    end;
  end;
end;

function TXMLDocHelper.AppendChild(Root: IXMLDomNode;
  ChildNodeName: string; AChildNodeValue: OleVariant): IXMLDomNode;
var
  AStr: string;
begin
  Result := AppendChild(Root, ChildNodeName);

  case  VarType( AChildNodeValue ) of

     varDate
            : if AChildNodeValue <> 0 then
                Result.nodeTypedValue := FormatDateTime( 'yyyy-mm-dd', AChildNodeValue);
     varEmpty,
     varNull : begin end;

     varSmallint,
     varInteger,
     varShortInt,
     varByte,
     varWord,
     varLongWord,
     varInt64
            : Result.nodeTypedValue := VarToStr( AChildNodeValue );
     varSingle,
     varDouble,
     varCurrency
            : begin
                AStr := CurrToStr(AChildNodeValue);
                if Pos( DecimalSeparator, AStr  ) > 0 then
                  AStr[Pos( DecimalSeparator, AStr  )] := '.';

                Result.nodeTypedValue := AStr;
              end;

     varStrArg,
     varString,
     varOleStr
           : Result.nodeTypedValue := VarToStr( AChildNodeValue );

     varBoolean
           : if AChildNodeValue = True then
                Result.nodeTypedValue := 'true'
             else
                Result.nodeTypedValue := 'false'

     else
       raise Exception.Create('AppendChild : VarType ' + VarTypeAsText(VarType(AChildNodeValue)) + ' not supported');
  end; //case
end;

function TXMLDocHelper.AppendChildNS(Root: IXMLDomNode; ChildNodeName,
  NamespaceURI: string): IXMLDomNode;
begin
  Result := Intf.createNode(NODE_ELEMENT, ChildNodeName, IfThen(NamespaceURI = '', Root.namespaceURI, NamespaceURI));
  Root.appendChild(Result);
end;

function CreateMSXML2DOMDocument: IDispatch;

  function TryCreate( AVersion: string ): IDispatch;
  begin
    Result := CreateOleObject(AVersion);
    _MSXML2_DOMDocument_Version := AVersion;
  end;

begin
  if _MSXML2_DOMDocument_Version <> '' then begin
    Result := CreateOleObject(_MSXML2_DOMDocument_Version);
    Exit;
  end;

  try
    Result := TryCreate('MSXML2.DOMDocument.6.0');
  except
    try
      Result := TryCreate('MSXML2.DOMDocument.5.0');
    except
      Result := CreateOleObject('MSXML2.DOMDocument.4.0');
    end;
  end;
end;

function CreateMSXML2XMLSchemaCache: IDispatch;

  function TryCreate( AVersion: string ): IDispatch;
  begin
    Result := CreateOleObject(AVersion);
    _MSXML2_XMLSchemaCache_Version := AVersion;
  end;

begin
  if _MSXML2_XMLSchemaCache_Version <> '' then begin
    Result := CreateOleObject(_MSXML2_XMLSchemaCache_Version);
    Exit;
  end;

  try
    Result := TryCreate('MSXML2.XMLSchemaCache.6.0');
  except
    try
      Result := TryCreate('MSXML2.XMLSchemaCache.5.0');
    except
      Result := CreateOleObject('MSXML2.XMLSchemaCache.4.0');
    end;
  end;
end;

initialization
 _MSXML2_DOMDocument_Version    := '';
 _MSXML2_XMLSchemaCache_Version := '';



end.

