unit WServUtils;

interface
uses
  SysUtils, Classes, WslService,
  WEBSUlil, InvokeRegistry, Types, SOAPAttach, SOAPAttachIntf,
  SHDocVw;

const
  PARAM_SEPARATOR_FOR_SIGN = ',';
  SIGN_TEST_MODE = 'test';
  WS_MAX_PART_SIZE = 100000;

  TRY_ATTEMPT_LATER = '    online- ';
  ERROR_TEXT = '     .     . ';

type
  TBeforeExec = procedure of object;

//     
  TActiveType = (BASE_ASSET,      //  
                 COLLATERAL_ASSET //  
                );

  TWebService = class
  private
    FWEBURL: string;
    FWslInterface: Wsl;
    FTestMode: boolean;
    FPersonCode: string;
    FOnBeforeExec: TBeforeExec;
    FMethodName: String;
    FWebBrowse: TWebbrowser;

    procedure PutPackageParts(PackageId, FileName: string);

    function GetPartSize(CurrentPartNumber, PartCount, FileSize: integer): integer;
    function GetPartCount(FileSize: integer): integer;
    Function GetPartPackage(Stream: TStream; const ABytes: Integer = MaxInt): TStream;
    function InitTransferIn (AFileName: string): data;
    function GetTransferResult(AData: WideString): outObj;
    procedure SetWEBURL(const AValue: string);

  public
    constructor Create;
    destructor Destroy; override;

    function GetOrderState(const DepositCode, RegNo: String): string;
    function GetRests(const DepositCode, SearchPersonCode, AccountCode, SectionCode, SecurityCode: string): string;
    function GetRestsRepo(const AccountCode, CorrSecTypeCode: string): string;
    function GetMarkedRests(const DepositCode, SearchPersonCode, AccountCode, SectionCode, SecurityCode: string;
                            ActiveType: TActiveType): string;
    function GetPackageList(const ADate: TDateTime): string;
    procedure GetPackage(const PackageId, FileName: string; FileSize: integer);
    procedure SendPackage(FileName: string);
    function GetRcCreditorAssets(const DebitorCode, CreditorCode, FiCode: string; RateNoMore: string): string;
    function GetCalcSuoPriceRests(const AccountCode, SecurityCode: string): string;
    procedure ConvertReposDoc(const AMode, ARepozitPersonCode, ASrcFileNameArh, ADstFileNameArh: string);

    property PersonCode: string read FPersonCode write FPersonCode;
    property TestMode: boolean read FTestMode write FTestMode;
    property OnBeforeExec: TBeforeExec read FOnBeforeExec write FOnBeforeExec;
    property WEBURL: string read FWEBURL write SetWEBURL;
  end;

  function WebService: TWebService;

implementation

uses
  StrUtils, Math, Variants, TypInfo, CriptU, SOAPHTTPClient;

var
  mWebService: TWebService;


function WebService: TWebService;
var
  URL : String;
begin
  //  -
  if not Assigned(mWebService) then begin
    mWebService := TWebService.Create;
  end;
  Result := mWebService;
  URL := MCryptForm.URLEdit.Text;
  if Trim(URL) <> '' then
  begin
    MCryptForm.LogMemo.Lines.Add('-> WS [WebService] URL=' + URL);
    Result.WEBURL := URL;
  end
  else
  begin
    MCryptForm.LogMemo.Lines.Add('-> WS [WebService] URL  ');
    Result := Nil;
    Exit;
  end;
  //  
   MCryptForm.acCryptoInit.Execute;
  //       TLS,
  //      
  if not Assigned(Result.FWebBrowse) then begin
    Result.FWebBrowse := TWebbrowser.Create(nil);
    Result.FWebBrowse.Navigate(Result.WEBURL);
  end;
end;

{ TWebService }

procedure TWebService.ConvertReposDoc(const AMode, ARepozitPersonCode, ASrcFileNameArh, ADstFileNameArh: string);
var
  InAttach: TSoapAttachment;
  OutAttach: InAttachment;
begin
  OutAttach := nil;
  try
    MCryptForm.LogMemo.Lines.Add('-> WS [ConvertReposDoc] ------------------');
    InAttach := TSoapAttachment.Create;
    try
      InAttach.SetSourceFile(ASrcFileNameArh);
      OutAttach := FWslInterface.ConvertReposDoc(AMode, InAttach, ARepozitPersonCode);
      if Assigned(OutAttach) then
        OutAttach.data.SaveToFile(ADstFileNameArh);
    finally
      FreeAndNil(InAttach);
      if Assigned(OutAttach) then
        FreeAndNil(OutAttach);
    end;
  except
    on E: FaultInfo do
      raise Exception.Create(E.errorDesc);
  end;
end;

constructor TWebService.Create;
var
  OwnCode : String;
begin
  FTestMode := False;
  MCryptForm.LogMemo.Lines.Add('-> WS [ConvertReposDoc] ------------------');
  OwnCode := MCryptForm.PCodEdit.Text;
  if Trim(OwnCode) <> '' then
  begin
    PersonCode:= OwnCode;
    MCryptForm.LogMemo.Lines.Add('-> WS [PersonCode] = ' + OwnCode);
  end;
end;

function TWebService.GetOrderState(const DepositCode, RegNo: String): string;
var
  os: data;
begin
  MCryptForm.LogMemo.Lines.Add('-> WS [GetOrderState] ------------------');
  try
    os := FWslInterface.GetOrderState(FPersonCode, DepositCode, RegNo);
    Result := os.data;
  finally
    if Assigned(os) then begin
      os.Free;
    end;
  end;
end;

procedure TWebService.GetPackage(const PackageId, FileName: string; FileSize: integer);
var
  p: InAttachment;
  partCount, partSize, i, l, j: integer;
  ss: TStringStream;
  fs: TStringStream;
  CopyString: string;
begin
  p := nil;
  FMethodName := 'GetPackage';
  try
    MCryptForm.LogMemo.Lines.Add('-> WS [GetPackage] ------------------');
    ss := TStringStream.Create;
    fs := TStringStream.Create;
    try
      partCount := GetPartCount(FileSize);
      for i := 1 to partCount do
      begin
        partSize := GetPartSize(i, partCount, FileSize);
        p := FWslInterface.GetPackage(FPersonCode, PackageId, i, partCount);
        ss.Clear;
        p.Data.SaveToStream(ss);
        CopyString := Copy(ss.DataString, 1, Length(ss.DataString) - 2);
        fs.WriteString(CopyString);
        DeleteFile(p.Data.CacheFile);
        if i = partCount then begin
          fs.SaveToFile(FileName);
        end;
      end;
    finally
      fs.Free;
      ss.Free;
      if Assigned(p) then
      begin
        p.Free;
      end;
    end;
  except
    on e: Exception do
    begin
      DeleteFile(p.Data.CacheFile);
      if e is FaultInfo then
      begin
        raise Exception.Create((e as FaultInfo).errorDesc);
      end
      else
      begin
        MCryptForm.LogMemo.Lines.Add('-> WS [GetPackage] Error: ' + e.Message);
        e.Message := ERROR_TEXT;
        raise;
      end;
    end;
  end;
end;

function TWebService.GetPackageList(const ADate: TDateTime): string;
var
  pl: data;
  dt: string;
begin
  pl := nil;
  FMethodName := 'GetPackage';
  try
    MCryptForm.LogMemo.Lines.Add('-> WS [GetPackageList] ------------------');
    try
      dt := FormatDateTime('dd.mm.yyy', ADate);
      pl := FWslInterface.GetPackageList(FPersonCode, dt);
      Result := pl.data;
    finally
      if Assigned(pl) then pl.Free;
    end;
  except
    on e: Exception do
    begin
      if e is FaultInfo then
      begin
        raise Exception.Create((e as FaultInfo).errorDesc);
      end
      else
      begin
        MCryptForm.LogMemo.Lines.Add('-> WS [GetPackageList] Error: ' + e.Message);
        e.Message := ERROR_TEXT;
        MCryptForm.LogMemo.Lines.Add('-> WS [GetPackageList] Error: ' + e.Message);
        raise;
      end;
    end;
  end;
end;

function TWebService.GetRests(const DepositCode, SearchPersonCode, AccountCode, SectionCode, SecurityCode: string): string;
var
  rs: data;
begin
  rs := nil;
  FMethodName := 'GetRests';
  try
    MCryptForm.LogMemo.Lines.Add('-> WS [GetRests] --------------------');

  {Result := '<?xml version="1.0" encoding="windows-1251"?>' +
            '<rests>' +
              '<rest>' +
                '<section_code>11RC0023430213000</section_code>' +
                '<section_type>11</section_type>' +
                '<section_state>fffffff</section_state>' +
                '<section_status>rrrrrrrr</section_status>' +
                '<security_code>SU26003RMFS2</security_code>' +
                '<security_name>ggggg</security_name>' +
                '<security_reg_num>26003RMFS</security_reg_num>' +
                '<value>10</value>' +
              '</rest>' +
              '<rest>' +
                '<section_code>11RC0023430213000</section_code>' +
                '<section_type>11</section_type>' +
                '<section_state>Открыт</section_state>' +
                '<section_status>Нет ограничений</section_status>' +
                '<security_code>SU26001RMFS6</security_code>' +
                '<security_name>ОФЗ ПК01</security_name>' +
                '<security_reg_num>26001RMFS</security_reg_num>' +
                '<value>30</value>' +
              '</rest>' +
              '<rest>' +
                '<section_code>11RC0023430213000</section_code>' +
                '<section_type>11</section_type>' +
                '<section_state>Открыт</section_state>' +
                '<section_status>Нет ограничений</section_status>' +
                '<security_code>SU26001RMFS6</security_code>' +
                '<security_name>ОФЗ ПК01</security_name>' +
                '<security_reg_num>26001RMFS</security_reg_num>' +
                '<value>20</value>' +
              '</rest>' +
              '<rest>' +
                '<section_code>11RC0023430213000</section_code>' +
                '<section_type>11</section_type>' +
                '<section_state>Открыт</section_state>' +
                '<section_status>Нет ограничений</section_status>' +
                '<security_code>SU26001RMFS6</security_code>' +
                '<security_name>ОФЗ ПК01</security_name>' +
                '<security_reg_num>26001RMFS</security_reg_num>' +
                '<value>5000575.00178</value>' +
              '</rest>' +
            '</rests>';}
    try
      rs := FWslInterface.GetRests(FPersonCode, DepositCode, SearchPersonCode, AccountCode, SectionCode, SecurityCode);
      Result := rs.data;
      MCryptForm.LogMemo.Lines.Add('-> WS GetRests: Result = ' + Result);
    finally
      if Assigned(rs) then rs.Free;
    end;
  except
    on e: Exception do
    begin
      if  e is FaultInfo then
      begin
        raise Exception.Create((e as FaultInfo).errorDesc);
      end
      else begin
        MCryptForm.LogMemo.Lines.Add('-> WS [GetRests] Error: ' + e.Message);
        e.Message := ERROR_TEXT;
        raise;
      end;
    end;
  end;
end;

function TWebService.GetTransferResult(AData: WideString): outObj;
begin
  FMethodName := 'GetTransferResult';
  try
    MCryptForm.LogMemo.Lines.Add('-> WS [GetTransferResult] -----------------');
    Result := FWslInterface.GetTransferResult(FPersonCode, AData);
  except
    on e: Exception do
    begin
      if  e is FaultInfo then
      begin
        MCryptForm.LogMemo.Lines.Add('-> WS [GetTransferResult] Error: ' + e.Message);
      end
      else
      begin
        MCryptForm.LogMemo.Lines.Add('-> WS [GetTransferResult] Error: ' + e.Message);
        e.Message := ERROR_TEXT;
        raise;
      end;
    end;
  end;
end;

function TWebService.InitTransferIn(AFileName: string): data;
begin
  FMethodName := 'InitTransferIn';
  try
    MCryptForm.LogMemo.Lines.Add('-> WS [InitTransferIn] -----------------');
    Result := FWslInterface.InitTransferIn(FPersonCode, AFileName);
  except
    on e: Exception do
    begin
      if  e is FaultInfo then
      begin
        MCryptForm.LogMemo.Lines.Add('-> WS [InitTransferIn] Error: ' + e.Message);
      end
      else
      begin
        MCryptForm.LogMemo.Lines.Add('-> WS [InitTransferIn] Error: ' + e.Message);
        e.Message := ERROR_TEXT;
        raise;
      end;
    end;
  end;
end;

destructor TWebService.Destroy;
begin
  FWslInterface := nil;
  if Assigned(FWebBrowse) then begin
    FreeAndNil(FWebBrowse);
  end;
  inherited;
end;

procedure TWebService.SendPackage(FileName: string);
var
  fn: string;
  ti: data;
  oo: outObj;
begin
  ti := nil;
  oo := nil;
  fn := ExtractFileName(FileName);
  try
    ti := InitTransferIn(fn);
    PutPackageParts(ti.data, FileName);
    oo := GetTransferResult(ti.data);
    if oo <> nil then
       MCryptForm.LogMemo.Lines.Add('-> WS [SendPackage] errorCode# ' + IntToStr(oo.errorCode) + ' ErrorDesc: ' + oo.errorDesc);
  finally
    if Assigned(ti) then ti.Free;
    if Assigned(oo) then oo.Free;
  end;
end;

procedure TWebService.SetWEBURL(const AValue: string);
begin
  MCryptForm.LogMemo.Lines.Add('-> WS [SetWEBURL] initialize, url: ' + AValue);
  if Assigned(FWslInterface) then
  begin
    if not AnsiSameText(FWEBURL, AValue) then
    begin
      FWslInterface := nil;
      FWEBURL := AValue;
      FWslInterface := GetWsl(FWEBURL);
    end;
  end
  else
  begin
    FWEBURL := AValue;
    FWslInterface := GetWsl(FWEBURL);
  end;
end;

procedure TWebService.PutPackageParts(PackageId, FileName: string);
var
  fs: TFileStream;
  SS: TStringStream;
  partCount, partSize, i: integer;
  FData: data;
  FSoapAttach: TSoapAttachment;
begin
  FData := nil;
  FMethodName := 'PutPackageParts';
  try
    MCryptForm.LogMemo.Lines.Add('-> WS [PutPackageParts] --------------- ');
    fs := TFileStream.Create(FileName, fmOpenRead);
    SS := TStringStream.Create;
    try
      partCount := GetPartCount(fs.Size);
      for i := 1 to partCount do
      begin
        partSize := GetPartSize(i, partCount, fs.Size);
    // create a soap attachment
        FSoapAttach := TSoapAttachment.Create;
        SS.Clear;
        SS.SetSize(partSize);
        SS.CopyFrom(fs, partSize);
        FSoapAttach.SetSourceStream(SS);
        FMethodName := 'PutPackage';
        MCryptForm.LogMemo.Lines.Add('-> WS [PutPackage] --------------- ');
        FData := FWslInterface.PutPackage(FPersonCode, PackageId, i, partCount, FSoapAttach);
      end;
    finally
      if Assigned(FData) then FData.Free;
      if Assigned(fs) then fs.Free;
      if Assigned(SS) then SS.Free;
    end;
  except
    on e: Exception do begin
      DeleteFile(FSoapAttach.CacheFile);
      if  e is FaultInfo then begin
        MCryptForm.LogMemo.Lines.Add('-> WS [PutPackage] Error: ' + e.Message);
      end
      else begin
        MCryptForm.LogMemo.Lines.Add('-> WS [PutPackage] Error: ' + e.Message);
        e.Message := ERROR_TEXT;
        raise;
      end;
    end;
  end;
end;

function TWebService.GetPartCount(FileSize: integer): integer;
begin
  Result := FileSize div WS_MAX_PART_SIZE + ord(FileSize mod WS_MAX_PART_SIZE <> 0);
end;

function TWebService.GetPartPackage(Stream: TStream;
  const ABytes: Integer): TStream;
var
  ms: TStream;
begin
  ms := TMemoryStream.Create;
  try
    ms.CopyFrom(Stream, min(ABytes, Stream.Size));
    ms.Position := 0;
    Result := ms;
  finally
    ms.Free;
  end;
end;

function TWebService.GetPartSize(CurrentPartNumber, PartCount, FileSize: integer): integer;
begin
  if CurrentPartNumber <> PartCount then
    Result := FileSize div PartCount
  else
    Result := FileSize - (FileSize div PartCount) * (PartCount - 1);
end;

function TWebService.GetRestsRepo(const AccountCode, CorrSecTypeCode: string): string;
var
  rs: data;
begin
{
    Result :=
      '<?xml version="1.0" encoding="UTF-8"?>' +
      '<RESTS>' +
      '<rest>' +
      '<security_code>security1</security_code>' +
      '<security_reg_num>reg_num1</security_reg_num>' +
      '<depo_acc_num>111111111111</depo_acc_num>' +
      '<section_num>11111111111111111</section_num>' +
      '<value>100</value>' +
      '<price>100</price>' +
      '</rest>' +
      '<rest>' +
      '<security_code>security1</security_code>' +
      '<security_reg_num>reg_num1</security_reg_num>' +
      '<depo_acc_num>111111111111</depo_acc_num>' +
      '<section_num>22222222222222222</section_num>' +
      '<value>50</value>' +
      '<price>1000</price>' +
      '</rest>' +
      ifthen(CorrSecTypeCode is '22', '',
      '<rest>' +
      '<security_code>security1</security_code>' +
      '<security_reg_num>reg_num1</security_reg_num>' +
      '<depo_acc_num>111111111111</depo_acc_num>' +
      '<section_num>33333333333333333</section_num>' +
      '<value>50</value>' +
      '<price>1000</price>' +
      '</rest>') +
      '</RESTS>'
}
  rs := nil;
  FMethodName := 'GetRestsRepo';
  try
    MCryptForm.LogMemo.Lines.Add('-> WS [GetRestsRepo] --------------- ');
    try
      rs := FWslInterface.GetRestsRepo(FPersonCode, AccountCode, CorrSecTypeCode);
      //CheckError(rs.faultcode, rs.faultstring);
      Result := rs.data;
    finally
      if Assigned(rs) then begin
        rs.Free;
      end;
    end;
  except
    on e: Exception do begin
      if  e is FaultInfo then begin
        raise Exception.Create((e as FaultInfo).errorDesc);
      end
      else begin
        MCryptForm.LogMemo.Lines.Add('-> WS [GetRestsRepo] Error: ' + e.Message);
        e.Message := ERROR_TEXT;
        raise;
      end;
    end;
  end;
end;

function TWebService.GetCalcSuoPriceRests(const AccountCode,
  SecurityCode: string): string;
var
  rs: data;
begin
  {Result := '<?xml version="1.0" encoding="windows-1251"?>' +
            '<assets>' +
              '<set>' +
                '<GA_CODE>22222222</GA_CODE>' +
                '<DEPO_ACC_NUM>222222</DEPO_ACC_NUM>' +
                '<SECTION_NUM>11RC0023430213000</SECTION_NUM>' +
                '<FI_NDC_CODE>SU26003RMFS2</FI_NDC_CODE>' +
                '<ISIN_CODE>26003RMFS</ISIN_CODE>' +
                '<FI_SHORT_NAME>SU26003RMFS2dddddddddddddddddddddd</FI_SHORT_NAME>' +
                '<AVAILABLE_REST>12</AVAILABLE_REST>' +
                '<DISCOUNT>26003RMFS</DISCOUNT>' +
                '<FI_PRICE>18</FI_PRICE>' +
                '<REST_PRICE>10</REST_PRICE>' +
              '</set>' +
              '<set>' +
                '<GA_CODE>1111111111</GA_CODE>' +
                '<DEPO_ACC_NUM>1111111111</DEPO_ACC_NUM>' +
                '<SECTION_NUM>11RC0023430213000</SECTION_NUM>' +
                '<FI_NDC_CODE>SU26003RMFS2</FI_NDC_CODE>' +
                '<ISIN_CODE>26003RMFS</ISIN_CODE>' +
                '<FI_SHORT_NAME>SU26003RMFS2dddddddddddddddddddddd</FI_SHORT_NAME>' +
                '<AVAILABLE_REST>12</AVAILABLE_REST>' +
                '<DISCOUNT>26003RMFS</DISCOUNT>' +
                '<FI_PRICE>18</FI_PRICE>' +
                '<REST_PRICE>1</REST_PRICE>' +
              '</set>' +
              '<set>' +
                '<GA_CODE>1111111111</GA_CODE>' +
                '<DEPO_ACC_NUM>1111111111</DEPO_ACC_NUM>' +
                '<SECTION_NUM>11RC0023430213000</SECTION_NUM>' +
                '<FI_NDC_CODE>SU26003RMFS2</FI_NDC_CODE>' +
                '<ISIN_CODE>26003RMFS</ISIN_CODE>' +
                '<FI_SHORT_NAME>SU26003RMFS2dddddddddddddddddddddd</FI_SHORT_NAME>' +
                '<AVAILABLE_REST>12</AVAILABLE_REST>' +
                '<DISCOUNT>26003RMFS</DISCOUNT>' +
                '<FI_PRICE>18</FI_PRICE>' +
                '<REST_PRICE>100</REST_PRICE>' +
              '</set>' +
              '<set>' +
                '<GA_CODE>1111111111</GA_CODE>' +
                '<DEPO_ACC_NUM>1111111111</DEPO_ACC_NUM>' +
                '<SECTION_NUM>11RC0023430213000</SECTION_NUM>' +
                '<FI_NDC_CODE>SU26003RMFS2</FI_NDC_CODE>' +
                '<ISIN_CODE>26003RMFS</ISIN_CODE>' +
                '<FI_SHORT_NAME>SU26003RMFS2dddddddddddddddddddddd</FI_SHORT_NAME>' +
                '<AVAILABLE_REST>12</AVAILABLE_REST>' +
                '<DISCOUNT>26003RMFS</DISCOUNT>' +
                '<FI_PRICE>18</FI_PRICE>' +
                '<REST_PRICE>2.1000</REST_PRICE>' +
              '</set>' +
            '</assets>'; }
  rs := nil;
  FMethodName := 'GetCalcSuoPriceRests';
  try
    MCryptForm.LogMemo.Lines.Add('-> WS [GetCalcSuoPriceRests] --------------- ');
    try
      rs := FWslInterface.GetSUOPrices(FPersonCode, AccountCode, SecurityCode);
      Result := rs.data;
      MCryptForm.LogMemo.Lines.Add('-> WS GetCalcSuoPriceRests Result: ' + Result);
    finally
      if Assigned(rs) then rs.Free;
    end;
  except
    on e: Exception do
    begin
      if  e is FaultInfo then
      begin
        raise Exception.Create((e as FaultInfo).errorDesc);
      end
      else
      begin
        MCryptForm.LogMemo.Lines.Add('-> WS [GetCalcSuoPriceRests] Error: ' + e.Message);
        e.Message := ERROR_TEXT;
        raise;
      end;
    end;
  end;
end;

function TWebService.GetMarkedRests(const DepositCode, SearchPersonCode, AccountCode, SectionCode, SecurityCode: string;
                                    ActiveType: TActiveType): string;
var
  rs: data;
  at: string;
begin
{
  Result := '<?xml version="1.0" encoding="windows-1251"?>' +
            '<rests>' +
              '<rest>' +
                '<section_code>11RC0023430213000</section_code>' +
                '<section_type>11</section_type>' +
                '<section_state>fffffff</section_state>' +
                '<section_status>rrrrrrrr</section_status>' +
                '<security_code>SU26003RMFS2</security_code>' +
                '<security_name>ggggg</security_name>' +
                '<security_reg_num>26003RMFS</security_reg_num>' +
                '<value>20</value>' +
                '<rest_cost>1</rest_cost>'+
              '</rest>' +
              '<rest>' +
                '<section_code>11RC0023430213000</section_code>' +
                '<section_type>11</section_type>' +
                '<section_state>Открыт</section_state>' +
                '<section_status>Нет ограничений</section_status>' +
                '<security_code>SU26001RMFS6</security_code>' +
                '<security_name>ОФЗ ПК01</security_name>' +
                '<security_reg_num>26001RMFS</security_reg_num>' +
                '<value>30</value>' +
                '<rest_cost>1</rest_cost>'+
              '</rest>' +
            '</rests>';
}
  rs := nil;
  FMethodName := 'GetMarkedRests';
  try
    MCryptForm.LogMemo.Lines.Add('-> WS [GetMarkedRests] --------------- ');
    try
      at := GetEnumName(TypeInfo(TActiveType), Integer(ActiveType));
      rs := FWslInterface.GetMarkedRests(
                 FPersonCode, DepositCode, SearchPersonCode, AccountCode, SectionCode, SecurityCode, at);
      Result := rs.data;
      MCryptForm.LogMemo.Lines.Add('-> WS GetMarkedRests Result: ' + Result);
    finally
      if Assigned(rs) then begin
        rs.Free;
      end;
    end;
  except
    on e: Exception do begin
      if  e is FaultInfo then begin
        raise Exception.Create((e as FaultInfo).errorDesc);
      end
      else
      begin
        MCryptForm.LogMemo.Lines.Add('-> WS [GetMarkedRests] Error: ' + e.Message);
        e.Message := ERROR_TEXT;
        raise;
      end;
    end;
  end;
end;

function TWebService.GetRcCreditorAssets(const DebitorCode, CreditorCode,
  FiCode: string; RateNoMore: string): string;
var
  rs: data;
begin
  {Result :=
      '<?xml version="1.0" encoding="windows-1251"?>' +
      '<assets>' +
      '<set>' +
      '<CREDITOR_FI_CODE>110VOZRP16</CREDITOR_FI_CODE>' +
      '<CREDITOR_FI_ISIN_CODE>RU0009100127</CREDITOR_FI_ISIN_CODE>' +
      '<CREDITOR_REST>1111</CREDITOR_REST>' +
      '<CREDITOR_CODE>PANORAM</CREDITOR_CODE>' +
      '<CREDITOR_SHORT_NAME></CREDITOR_SHORT_NAME>' +
      '<CREDITOR_LIMIT_PRICE>200</CREDITOR_LIMIT_PRICE>' +
      '<CREDITOR_LIMIT_REST>300</CREDITOR_LIMIT_REST>' +
      '<CREDITOR_RATE_VALUE>300</CREDITOR_RATE_VALUE>' +
      '<CREDITOR_FI_PRICE>400</CREDITOR_FI_PRICE>' +
      '<CREDITOR_FI_VALUE>111,11111111119</CREDITOR_FI_VALUE>' +
      '<CREDITOR_FI_PRICE_RATE>600</CREDITOR_FI_PRICE_RATE>' +
      '<CREDITOR_FI_VALUE_RATE>700</CREDITOR_FI_VALUE_RATE>' +
      '</set>' +
      '<set>' +
      '<CREDITOR_FI_CODE>B-GUTI/01-01</CREDITOR_FI_CODE>' +
      '<CREDITOR_FI_ISIN_CODE>RU0007202495</CREDITOR_FI_ISIN_CODE>' +
      '<CREDITOR_REST>2222</CREDITOR_REST>' +
      '<CREDITOR_CODE>NDC000000000</CREDITOR_CODE>' +
      '<CREDITOR_SHORT_NAME>  </CREDITOR_SHORT_NAME>' +
      '<CREDITOR_LIMIT_PRICE>201</CREDITOR_LIMIT_PRICE>' +
      '<CREDITOR_LIMIT_REST>302</CREDITOR_LIMIT_REST>' +
      '<CREDITOR_RATE_VALUE>303</CREDITOR_RATE_VALUE>' +
      '<CREDITOR_FI_PRICE>404</CREDITOR_FI_PRICE>' +
      '<CREDITOR_FI_VALUE>0</CREDITOR_FI_VALUE>' +
      '<CREDITOR_FI_PRICE_RATE>606</CREDITOR_FI_PRICE_RATE>' +
      '<CREDITOR_FI_VALUE_RATE>707</CREDITOR_FI_VALUE_RATE>' +
      '</set>' +
      '<set>' +
      '<CREDITOR_FI_CODE>FINP</CREDITOR_FI_CODE>' +
      '<CREDITOR_FI_ISIN_CODE>RU0008117536</CREDITOR_FI_ISIN_CODE>' +
      '<CREDITOR_REST>3333</CREDITOR_REST>' +
      '<CREDITOR_CODE>MC0124400000</CREDITOR_CODE>' +
      '<CREDITOR_SHORT_NAME> "" ()</CREDITOR_SHORT_NAME>' +
      '<CREDITOR_LIMIT_PRICE>200</CREDITOR_LIMIT_PRICE>' +
      '<CREDITOR_LIMIT_REST>300</CREDITOR_LIMIT_REST>' +
      '<CREDITOR_RATE_VALUE>300</CREDITOR_RATE_VALUE>' +
      '<CREDITOR_FI_PRICE>400</CREDITOR_FI_PRICE>' +
      '<CREDITOR_FI_VALUE>0</CREDITOR_FI_VALUE>' +
      '<CREDITOR_FI_PRICE_RATE>600</CREDITOR_FI_PRICE_RATE>' +
      '<CREDITOR_FI_VALUE_RATE>700</CREDITOR_FI_VALUE_RATE>' +
      '</set>' +
      '</assets>';}
  rs := nil;
  FMethodName := 'GetRcCreditorAssets';
  try
    MCryptForm.LogMemo.Lines.Add('-> WS [GetRcCreditorAssets] --------------- ');
    try
      rs := FWslInterface.GetRcCreditorAssets(FPersonCode, DebitorCode, CreditorCode, FiCode, RateNoMore);
      Result := rs.data;
      MCryptForm.LogMemo.Lines.Add('-> WS GetRcCreditorAssets Result: ' + Result);
    finally
      if Assigned(rs) then rs.Free;
    end;
  except
    on e: Exception do
    begin
      if  e is FaultInfo then
      begin
        raise Exception.Create((e as FaultInfo).errorDesc);
      end
      else
      begin
        MCryptForm.LogMemo.Lines.Add('-> WS [GetRcCreditorAssets] Error: ' + e.Message);
        e.Message := ERROR_TEXT;
        raise;
      end;
    end;
  end;
end;
end.
