unit WEBSUlil;

interface
uses SOAPHTTPClient, Classes, Windows, XMLUtil, MSXML2_TLB, SysUtils, IdCoderMIME,
      Math, StrUtils, Variants, Dialogs, SOAPAttach, InvokeRegistry, WSDLIntf, ActiveX, ComObj,
      XMLIntf, SOAPAttachIntf, OPToSOAPDomConv, IntfInfo, XMLDoc, DateUtils;

type
  TDoSignEvent = procedure(Value: AnsiString; SignStream: TStream) of object;
  TXmlCanonicalizer = class
  private
    function BuildXmlString(ARootNode: IXMLDOMNode): string;
    function BuildAttributes(ANode: IXMLDOMNode): string;
    function NormalizeAttributeValue(AValue: string): string;
    function NormalizeText(AText: string): string;
  public
    function Canonicalize(ARootNode: IXMLDOMNode): string;
  end;
  TSoap = class
  private
    FOnDoSign: TDoSignEvent;
    FEncoder: TIdEncoderMIME;
    FDecoder: TIdDecoderMIME;
    FSignReferences: TStrings;
    FEenvNameSpace: string;
    FXMLDomDocument: IXMLDOMDocument;
    Canonicalizer: TXmlCanonicalizer;
    function EncodeStream(Stream: TStream; const ABytes: Integer = MaxInt): string;
    procedure DecodeStream(Stream: TStream; CodedFileBody: string);
    function GetSign(strForSign: string): string; overload;
    function GetSign(Params: array of variant): string; overload;
    function GetNameSpace(ANode: IXMLDOMNode): string;
    function CreateSignedInfo(ADom: IXMLDomDocument; const ANameSpace: string): IXMLDomNode;
    procedure SetSignReferences(const Value: TStrings);
    function CreateSignature(ASignedInfo: IXMLDomNode; ANameSpace: string): IXMLDomNode; virtual;
    function CreateSecuredKeyInfo(ASignature: IXMLDOMNode; const ANameSpace: string): IXMLDOMNode;
    procedure Checks(Astream: TStream);
  public
    property XmlDomDocument: IXMLDOMDocument read FXMLDomDocument;
    constructor Create; //override;
    destructor Destroy;
    property OnDoSign: TDoSignEvent read FOnDoSign write FOnDoSign;
    function BuildSoapRequest(AXml: string): string;
  published
    property SignReferences: TStrings read FSignReferences write SetSignReferences;
  end;

  TNSDOPToSoapDomConvert = class (TOPToSoapDomConvert)
  private
    FSoap: TSoap;
  public
    property Soap: TSoap read FSoap;
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    function  InvContextToMsg(const IntfMD: TIntfMetaData;
                              MethNum: Integer;
                              Con: TInvContext;
                              Headers: THeaderList): TStream; override;
    procedure ProcessResponse(const Resp: TStream;
                              const IntfMD: TIntfMetaData;
                              const MD: TIntfMethEntry;
                              Context: TInvContext;
                              Headers: THeaderList); override;
  end;

  TNSDHTTPRIO = class(THTTPRIO)
  private
    FConverter: TNSDOPToSoapDomConvert;
  protected
    procedure DoAfterExecute(const MethodName: string; Response: TStream); override;
    procedure DoBeforeExecute(const MethodName: string; Request: TStream); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  end;

implementation

uses
  CriptU;

{ TNSDHTTPRIO }
var
    FHeader: IXMLDOMNode;
    Flag: boolean;
const
  ENTITY_NAMES: array[1..3] of string =  ('&amp;', '&gt;', '&lt;');
  ENTITY_CHARS: array[1..3] of string =  ('&',     '>',    '<');

function CharToEntityName(Value: string): string;
var
  i: integer;
begin
  Result := Value;
  for i := Low(ENTITY_NAMES) to High(ENTITY_NAMES) do Result := ReplaceStr(Result, ENTITY_CHARS[i], ENTITY_NAMES[i]);
end;

function StreamToString(AStream: TStream): string;
var
  SS: TStringStream;
begin
  try
    SS := TStringStream.Create;
    SS.CopyFrom(AStream, AStream.Size - AStream.Position);
    Result := SS.DataString;
  finally
    FreeAndNil(SS);
  end;
end;

function TSoap.BuildSoapRequest(AXml: string): string;
var
  Envelope, Header, Security,
  SignedInfo, Signature: IXMLDOMNode;
begin
  XmlDomDocument.LoadXML(AXml);
  SignReferences.Clear;
  SignReferences.Add('NRDRequest');
  FEenvNameSpace := GetNameSpace(XmlDomDocument.LastChild);
  Assert(FEenvNameSpace <> '');
  Header := XmlDomDocument.SelectSingleNode('//' + FEenvNameSpace + ':Header');
  if (Header = nil) then
  begin
    Envelope := XmlDomDocument.SelectSingleNode('//' + FEenvNameSpace + ':Envelope');
    Assert(Envelope <> nil);
    Header := XmlDomDocument.CreateElement(FEenvNameSpace + ':Header');
    Envelope.InsertBefore(Header, Envelope.FirstChild);
  end;
  SignedInfo := CreateSignedInfo(XmlDomDocument, 'ds');
  Signature := CreateSignature(SignedInfo, 'ds');
  Security := CreateSecuredKeyInfo(Signature, 'wsse');
  Header.InsertBefore(Security, Header.FirstChild);
  FHeader := Header;
  Result := XmlDomDocument.xml;
end;

constructor TSoap.Create;
begin
  FOnDoSign := MCryptForm.SignStringToStream;
  FSignReferences := TStringList.Create();
  FXMLDomDocument := CoDOMDocument.Create();
  Canonicalizer := TXmlCanonicalizer.Create();
end;

function TSoap.CreateSecuredKeyInfo(ASignature: IXMLDOMNode;
  const ANameSpace: string): IXMLDOMNode;
var
  keyInfo, node, reference: IXMLDOMNode;
  certValue, encodedCertValue: string;
begin
  Result := ASignature.ownerDocument.createElement(ANameSpace + ':Security');
  (Result as IXMLDomElement).setAttribute('xmlns:' + ANameSpace, 'http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd');
  Result.appendChild(ASignature);
end;

function TSoap.CreateSignature(ASignedInfo: IXMLDomNode;
  ANameSpace: string): IXMLDomNode;
var
  node: IXMLDomNode;
  sigValue: string;

begin
  Result := ASignedInfo.ownerDocument.createElement(ANameSpace + ':Signature');
  Result.appendChild(ASignedInfo);
  (Result as IXMLDomElement).setAttribute('xmlns:' + ANameSpace, 'http://www.w3.org/2000/09/xmldsig#');
  node := ASignedInfo.ownerDocument.createElement(ANameSpace + ':SignatureValue');
  Result.appendChild(node);
  node.text := GetSign(canonicalizer.Canonicalize(ASignedInfo));
end;

function TSoap.CreateSignedInfo(ADom: IXMLDomDocument;
  const ANameSpace: string): IXMLDomNode;
var
  i: Integer;
  reference, data, node, tmpNode: IXMLDomNode;
  stream: TMemorystream;
begin
  Result := ADom.createElement(ANameSpace + ':SignedInfo');
  (Result as IXMLDomElement).setAttribute('xmlns:' + ANameSpace, 'http://www.w3.org/2000/09/xmldsig#');
  node := ADom.createElement(ANameSpace + ':CanonicalizationMethod');
  Result.appendChild(node);
  (node as IXMLDomElement).setAttribute('Algorithm', 'http://www.w3.org/2001/10/xml-exc-c14n#');

  node := ADom.createElement(ANameSpace + ':SignatureMethod');
  Result.appendChild(node);
  (node as IXMLDomElement).setAttribute('Algorithm', 'http://www.w3.org/2001/04/xmldsig-more#gostr34102001-gostr3411');

  for i := 0 to SignReferences.Count - 1 do
  begin
    reference := ADom.createElement(ANameSpace + ':Reference');
    Result.appendChild(reference);
    (reference as IXMLDomElement).setAttribute('URI', '#' + SignReferences[i]);
    node := ADom.selectSingleNode('//' + FEenvNameSpace + ':Body');
    (node as IXMLDomElement).setAttribute('wsu:Id', 'NRDRequest');
    (node as IXMLDomElement).setAttribute('xmlns:wsu', 'http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd');
    (node as IXMLDomElement).setAttribute('xmlns:' + FEenvNameSpace, 'http://schemas.xmlsoap.org/soap/envelope/');

    data := ADom.selectSingleNode('//*[@wsu:Id="' + SignReferences[i] + '"]');
    Assert(data <> nil);

    node := ADom.createElement(ANameSpace + ':Transforms');
    tmpNode := ADom.createElement(ANameSpace + ':Transform');
    node.appendChild(tmpNode);
    (tmpNode as IXMLDomElement).setAttribute('Algorithm', 'http://www.w3.org/2001/10/xml-exc-c14n#');
    reference.appendChild(node);

    node := ADom.createElement(ANameSpace + ':DigestMethod');
    reference.appendChild(node);
    (node as IXMLDomElement).setAttribute('Algorithm', 'http://www.w3.org/2001/04/xmldsig-more#gostr3411');

    node := ADom.createElement(ANameSpace + ':DigestValue');
    reference.appendChild(node);
    try
      stream := TMemorystream.Create;
      MCryptForm.GetMemHashStream(canonicalizer.Canonicalize(data), stream);
      node.text := EncodeStream(stream);
    finally
      FreeAndNil(stream);
    end;
  end;
end;

procedure TSoap.DecodeStream(Stream: TStream; CodedFileBody: string);
begin
  if not Assigned(FDecoder) then
    FDecoder := TIdDecoderMIME.Create(nil);
  FDecoder.DecodeBegin(Stream);
  try
    FDecoder.Decode(CodedFileBody);
  finally
    FDecoder.DecodeEnd;
  end;
end;

destructor TSoap.Destroy;
begin
  FreeAndNil(FOnDoSign);
  FreeAndNil(FSignReferences);
  FreeAndNil(FXmlDomDocument);
  Canonicalizer.Free();
end;

constructor TNSDHTTPRIO.Create(AOwner: TComponent);
begin
  inherited;
  FConverter := TNSDOPToSoapDomConvert.Create(nil);
  Converter := FConverter;
end;

destructor TNSDHTTPRIO.Destroy;
begin
  inherited;
end;

procedure TNSDHTTPRIO.DoAfterExecute(const MethodName: string;
  Response: TStream);
begin
  if Response.Size = 0 then begin
    raise Exception.Create('  .   !');
  end;
  with TStringStream.Create('') do
  try
    CopyFrom(Response, Response.Size - Response.Position);
    MCryptForm.LogMemo.Lines.Add('-> WS AfterExecute: ' + DataString);
  finally
    Free;
  end;
  inherited;
end;

procedure TNSDHTTPRIO.DoBeforeExecute(const MethodName: string;
  Request: TStream);
begin
  inherited;
  with TStringStream.Create('') do
  try
    CopyFrom(Request, Request.Size - Request.Position);
    MCryptForm.LogMemo.Lines.Add('-> WS BeforeExecute: ======================>');
    MCryptForm.LogMemo.Lines.Add(DataString);
  finally
      Free;
  end;
end;

function TSoap.EncodeStream(Stream: TStream;
  const ABytes: Integer): string;
var
  ms: TMemoryStream;
begin
  {if not Assigned(FEncoder) then}
  FEncoder := TIdEncoderMIME.Create(nil);
  ms := TMemoryStream.Create;
  try
    ms.CopyFrom(Stream, min(ABytes, Stream.Size - Stream.Position));
    ms.Position := 0;
    Result := FEncoder.Encode(ms);
  finally
    FreeAndNil(ms);
    FreeAndNil(FEncoder);
  end;
end;

function TSoap.GetSign(strForSign: string): string;
var
  i: integer;
  stream: TStream;
begin
  if not Assigned(FOnDoSign) then begin
    raise Exception.Create('Crypto not initialized');
  end else begin
    stream := TMemoryStream.Create;
    try
      FOnDoSign(AnsiString(strForSign), stream);
      stream.Position := 0;
      Result := EncodeStream(stream);
    finally
      FreeAndNil(stream);
    end;
  end;
end;

function TSoap.GetNameSpace(ANode: IXMLDOMNode): string;
var
  ind: Integer;
begin
  ind := Pos(':', ANode.nodeName);
  if (ind > 0) then
  begin
    Result := Copy(ANode.nodeName, 1, ind - 1);
  end else
  begin
    Result := '';
  end;
end;

function TSoap.GetSign(Params: array of variant): string;
var
  i: integer;
  strForSign: string;
  stream: TStream;
begin
  strForSign := '';
  for i := 0 to Length(Params) - 1 do begin
    strForSign := strForSign + ',' + VarToStr(Params[i]);
  end;
  strForSign := AnsiRightStr(strForSign, Length(strForSign) - 1);

  if not Assigned(FOnDoSign) then begin
    raise Exception.Create('Crypto not initialized');
  end else begin
    stream := TMemoryStream.Create;
    try
      FOnDoSign(AnsiString(strForSign), stream);
      stream.Position := 0;
      Result := EncodeStream(stream);
    finally
      FreeAndNil(stream);
    end;
  end;
end;

procedure TSoap.SetSignReferences(const Value: TStrings);
begin
  FSignReferences.Assign(Value);
end;

{ TXmlCanonicalizer }

function TXmlCanonicalizer.BuildAttributes(ANode: IXMLDOMNode): string;
var
   i: Integer;
   attributes, namespaces: TStringList;
   element: IXMLDOMElement;
begin
  Result := '';
  if not Supports(ANode, IXMLDOMElement) then Exit;
  attributes := nil;
  namespaces := nil;
  try
    attributes := TStringList.Create();
    attributes.Sorted := True;

    namespaces := TStringList.Create();
    namespaces.Sorted := True;

    element := (ANode as IXMLDOMElement);
    for i := 0 to element.attributes.length - 1 do
    begin
       if (system.Pos('xmlns', LowerCase(element.attributes.item[i].nodeName)) = 1) then
       begin
          namespaces.Add(element.attributes.item[i].nodeName + '="' + NormalizeAttributeValue(VarToStr(element.attributes.item[i].nodeValue)) + '"');
       end else
       begin
          attributes.Add(element.attributes.item[i].nodeName + '="' + NormalizeAttributeValue(VarToStr(element.attributes.item[i].nodeValue)) + '"');
       end;
    end;

    for i := 0 to namespaces.Count - 1 do
    begin
       Result := Result + ' ' + Trim(namespaces[i]);
    end;

    for i := 0 to attributes.Count - 1 do
    begin
       Result := Result + ' ' + Trim(attributes[i]);
    end;
  finally
    namespaces.Free();
    attributes.Free();
  end;
end;

function TXmlCanonicalizer.BuildXmlString(ARootNode: IXMLDOMNode): string;
var
   i: Integer;
begin
  if Supports(ARootNode, IXMLDOMText) then
  begin
    Result := Result + VarToStr(ARootNode.nodeValue);
  end else
  begin
    Result := '<' + ARootNode.nodeName + BuildAttributes(ARootNode) + '>';
    for i := 0 to ARootNode.childNodes.length - 1 do
    begin
       Result := Result + BuildXmlString(ARootNode.childNodes.item[i]);
    end;
    Result := Result + '</' + ARootNode.nodeName + '>';
  end;
end;

function TXmlCanonicalizer.Canonicalize(ARootNode: IXMLDOMNode): string;
begin
  Result := AnsiToUtf8(BuildXmlString(ARootNode));
end;

function TXmlCanonicalizer.NormalizeAttributeValue(AValue: string): string;
begin
  Result := AValue;
  Result := StringReplace(Result, '"', '&quot;', [rfReplaceAll]);
  Result := StringReplace(Result, #9, #32, [rfReplaceAll]);
  Result := StringReplace(Result, #13#10, #32, [rfReplaceAll]);
  Result := StringReplace(Result, #13, #32, [rfReplaceAll]);
  Result := StringReplace(Result, #10, #32, [rfReplaceAll]);
end;

function TXmlCanonicalizer.NormalizeText(AText: string): string;
begin
  Result := AText;
  Result := StringReplace(Result, #13#10, #10, [rfReplaceAll]);
  Result := StringReplace(Result, #13, #10, [rfReplaceAll]);
end;

procedure TSoap.Checks(Astream: TStream);
var
  Envelope, Header, Security, Signature, Data,
  DigestValue, SignatureValue, Body, SignedInfo,
  ErrorCode, FaultCode, Fault: IXMLDOMNode;
  DsNameSpace: string;
  Stream: TMemorystream;
  StringStream: TStringStream;
begin
  XmlDomDocument.validateOnParse := true;
  XmlDomDocument.loadXML(StreamToString(Astream));

  if not Assigned(XmlDomDocument.lastChild) then begin
    raise Exception.Create('      ');
  end;
  FEenvNameSpace := GetNameSpace(XmlDomDocument.LastChild);
  Assert(FEenvNameSpace <> '');
  Body := XmlDomDocument.SelectSingleNode('//' + FEenvNameSpace + ':Body');
  if not Assigned(Body) then begin
    raise Exception.Create('       Body');
  end;
  Fault := Body.firstChild;
  if not Assigned(Fault) then begin
    Header := XmlDomDocument.SelectSingleNode('//' + FEenvNameSpace + ':Header');
    if not Assigned(Header) then begin
      raise Exception.Create('       Header');
    end;
    Security := Header.firstChild;
    Signature := Security.firstChild;
    DsNameSpace := GetNameSpace(Signature);

    //   
    SignedInfo := XmlDomDocument.SelectSingleNode('//' + DsNameSpace + ':SignedInfo');
    if not Assigned(SignedInfo) then begin
      raise Exception.Create('       SignedInfo');
    end;

    if not Assigned((SignedInfo as IXMLDomElement).getAttributeNode('xmlns:' + DsNameSpace)) then begin
      (SignedInfo as IXMLDomElement).setAttribute('xmlns:' + DsNameSpace, 'http://www.w3.org/2000/09/xmldsig#');
    end;

    DigestValue := XmlDomDocument.SelectSingleNode('//' + DsNameSpace + ':DigestValue');
    if not Assigned(DigestValue) then begin
      raise Exception.Create('       DigestValue');
    end;

    SignatureValue := XmlDomDocument.SelectSingleNode('//' + DsNameSpace + ':SignatureValue');
    if not Assigned(SignatureValue) then begin
      raise Exception.Create('   c    SignatureValue');
    end;

    Body := XmlDomDocument.SelectSingleNode('//' + FEenvNameSpace + ':Body');
    if not Assigned(Body) then begin
      raise Exception.Create('       Body');
    end;

    if not Assigned((Body as IXMLDomElement).getAttributeNode('xmlns:' + FEenvNameSpace)) then begin
      (Body as IXMLDomElement).setAttribute('xmlns:' + FEenvNameSpace, 'http://schemas.xmlsoap.org/soap/envelope/');
    end;

    Data := XmlDomDocument.SelectSingleNode('//data');
    if Assigned(Data) then begin
      if Assigned((Data as IXMLDomElement).getAttributeNode('xmlns:xsd')) then begin
        (Data as IXMLDomElement).removeAttribute('xmlns:xsd');
      end;
      Data.text := CharToEntityName(Data.text);
    end;
    try
      stream := TMemorystream.Create;
      MCryptForm.GetMemHashStream(Canonicalizer.Canonicalize(Body), stream);
      Stream.Position := 0;
      if DigestValue.text <> EncodeStream(stream) then begin
        raise Exception.Create('      ');
      end;
    finally
      FreeAndNil(stream);
    end;
    try
      StringStream := TStringStream.Create;
      DecodeStream(StringStream, SignatureValue.text);
    finally
      StringStream.Free;
    end;
  end;
end;

{ TNSDOPToSoapDomConvert }

constructor TNSDOPToSoapDomConvert.Create(AOwner: TComponent);
begin
  inherited;
  FSoap := TSoap.Create;
end;

destructor TNSDOPToSoapDomConvert.Destroy;
begin
  FreeAndNil(FSoap);
  inherited;
end;

function TNSDOPToSoapDomConvert.InvContextToMsg(const IntfMD: TIntfMetaData;
  MethNum: Integer; Con: TInvContext; Headers: THeaderList): TStream;
var
  SoapRequestStr: string;
  SS: TStringStream;
begin
  Result := inherited;
  try
    SS := TStringStream.Create;
    SS.CopyFrom(Result, Result.Size - Result.Position);
    SoapRequestStr := Soap.BuildSoapRequest(SS.DataString);
    SS.Clear;
    SS.WriteString(SoapRequestStr);
    Result.Position := 0;
    Result.CopyFrom(SS, SS.Size - SS.Position);
    Result.Position := 0;
  finally
    FreeAndNil(SS);
  end;
end;

procedure TNSDOPToSoapDomConvert.ProcessResponse(const Resp: TStream;
  const IntfMD: TIntfMetaData; const MD: TIntfMethEntry; Context: TInvContext;
  Headers: THeaderList);
var
  ResponseString: string;
begin
  Soap.Checks(Resp);
  Resp.Position := 0;
  inherited;
end;

end.
