package inist.nsdws.soap;

import inist.nsdws.common.ClientBase;
import inist.nsdws.common.NSDException;
import inist.nsdws.soap.gen.*;
import org.apache.cxf.attachment.AttachmentUtil;
import org.apache.cxf.endpoint.Client;
import org.apache.cxf.frontend.ClientProxy;
import org.apache.cxf.helpers.IOUtils;
import org.apache.cxf.message.Attachment;
import org.apache.cxf.message.Message;
import org.apache.cxf.transport.http.HTTPConduit;
import org.apache.cxf.transports.http.configuration.HTTPClientPolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.activation.DataHandler;
import javax.mail.internet.InternetHeaders;
import javax.xml.ws.BindingProvider;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.lang.Exception;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * Description: NSD SOAP Web-service client
 * User: Jury Rib, Inist
 * Date: 08.11.13
 * Time: 16:15
 */
public class ClientSOAP extends ClientBase {
    final Logger logger = LoggerFactory.getLogger(ClientSOAP.class);

    private final String nsdMemberCode;
    private String serviceEndPointURL;


    private Wsl createPort(){
        WslService ss = new WslService();
        Wsl port = ss.getWslPort();
        if(serviceEndPointURL!=null){
            Map<String, Object> ctx = ((BindingProvider) port).getRequestContext();
            ctx.put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, serviceEndPointURL);
        }
        Client client = ClientProxy.getClient(port);
        HTTPConduit http = (HTTPConduit) client.getConduit();
        http.setTlsClientParameters(TlsUtil.getTlsParams());

        HTTPClientPolicy httpClientPolicy = new HTTPClientPolicy();
        httpClientPolicy.setAllowChunking(false);
        httpClientPolicy.setConnectionTimeout(360000);
        httpClientPolicy.setReceiveTimeout(360000);
        http.setClient(httpClientPolicy);
        return port;
    }

    /**
     * NSD SOAP Web-service client constructor
     */
    public ClientSOAP() throws Exception{
        logger.info("Start SOAP WS client initialization");

        //check NSD member code
        nsdMemberCode = properties.getProperty("nsd.memberCode");
        if (nsdMemberCode == null)
            throw new IllegalArgumentException("Не задан код участника НРД");

        //service endpoint URL
        serviceEndPointURL = properties.getProperty("soap.service.URI");

        //check keystore path
        String keystorePath = properties.getProperty("keystore.path");
        if (keystorePath == null) throw new IllegalArgumentException("Не задан путь к хранилищу ключей");

        //check keystore password
        String keystorePassword = properties.getProperty("keystore.password");
        if (keystorePassword == null) throw new IllegalArgumentException("Не задан пароль доступа к хранилищу ключей");

        //check private key alias
        String privateKeyAlias = properties.getProperty("privateKey.alias");
        if (privateKeyAlias == null)
            throw new IllegalArgumentException("Не задан код (alias) первичного ключа в хранилище");

        //init SSL
        TlsUtil.initializeConduitForSSL(properties);

        logger.info("SOAP WS client initialized");
    }


    /**
     * Documents package send
     *
     * @param file package file
     * @throws Exception
     */
    public void sendPackage(File file) throws Exception {
        Wsl port = createPort();
        //initTransferIn
        String packageId;
        try {
            logger.debug("InitTransferIn started");
            StringResult result = port.initTransferIn(nsdMemberCode, file.getName());
            packageId = result.getData();
            System.out.println("InitTransferIn: PackageID=" + packageId);
            logger.debug("InitTransferIn finished");
        } catch (WsFault e) {
            logger.error("InitTransferIn error", e);
            FaultInfo faultInfo = e.getFaultInfo();
            if (faultInfo != null) {
                throw new NSDException(faultInfo.getErrorCode(), faultInfo.getErrorDesc(), e);
            }
            throw e;
        }

        //PutPackage
        try {
            String id = "package1";
            List<Attachment> attachments = new ArrayList<>();
            InternetHeaders headers = new InternetHeaders();
            headers.addHeader("Content-Type", "application/binary");
            headers.addHeader("Content-ID", id);
            headers.addHeader("Content-Transfer-Encoding", "binary");
            byte[] bb = IOUtils.readBytesFromStream(new FileInputStream(file));
            ByteArrayInputStream bais = new ByteArrayInputStream(bb);
            Attachment attach = AttachmentUtil.createAttachment(bais, headers);
            attachments.add(attach);
            BindingProvider bp = (BindingProvider) port;
            java.util.Map<String, Object> reqContext = bp.getRequestContext();
            reqContext.put(Message.ATTACHMENTS, attachments);

            logger.debug("PutPackage started");
            PutPackage.PackageBody packageBody = new PutPackage.PackageBody();
            packageBody.setHref("cid:"+id);
            port.putPackage(nsdMemberCode, packageId, 1, 1, packageBody);
            logger.debug("PutPackage finished");
        } catch (WsFault e) {
            logger.error("PutPackage error", e);
            FaultInfo faultInfo = e.getFaultInfo();
            if (faultInfo != null) {
                throw new NSDException(faultInfo.getErrorCode(), faultInfo.getErrorDesc(), e);
            }
            throw e;
        }
        //GetTransferResult
        try {
            logger.debug("GetTransferResult started");
            port.getTransferResult(nsdMemberCode, packageId);
            logger.debug("GetTransferResult finished");
        } catch (WsFault e) {
            logger.error("GetTransferResult error", e);
            FaultInfo faultInfo = e.getFaultInfo();
            if (faultInfo != null) {
                throw new NSDException(faultInfo.getErrorCode(), faultInfo.getErrorDesc(), e);
            }
            throw e;
        }
    }

    /**
     * retrieve list of available packages for client
     * @param date date of interest
     * @return list of available packages in xml format
     * @throws Exception
     */
    public String getPackageList(Date date) throws Exception {
        Wsl port = createPort();

        String packageListXML;
        try {
            logger.debug("GetPackageList started");
            String dateStr = new SimpleDateFormat("dd.MM.yyyy").format(date);
            StringResult result = port.getPackageList(nsdMemberCode, dateStr);
            packageListXML = result.getData();
            System.out.println("GetPackageList: packageListXML=" + packageListXML);
            logger.debug("GetPackageList finished");
        } catch (WsFault e) {
            logger.error("GetPackageList error", e);
            FaultInfo faultInfo = e.getFaultInfo();
            if (faultInfo != null) {
                throw new NSDException(faultInfo.getErrorCode(), faultInfo.getErrorDesc(), e);
            }
            throw e;
        }
        return packageListXML;
    }

    /**
     * retrieve one package from server
     * @param packageId package idenifier
     * @param fileName name of package file
     * @param directoryOut directory to save file in
     * @throws Exception
     */
    public void getPackage(String packageId, String fileName, File directoryOut) throws Exception {

        if (directoryOut != null && !directoryOut.exists()) {
            if (!directoryOut.mkdir())
                throw new NSDException("Can't create output directory: " + directoryOut.getAbsolutePath());
        }
        Wsl port = createPort();

        try {
            logger.debug("GetPackage started. PackageId=" + packageId);
            ByteResult result = port.getPackage(nsdMemberCode, packageId, 1, 1);
            result.getData();
            BindingProvider bp = (BindingProvider) port;
            java.util.Map<String, Object> respContext = bp.getResponseContext();
            @SuppressWarnings("unchecked")
            Collection<Attachment> attachments = (Collection<Attachment>) respContext.get(Message.ATTACHMENTS);
            if (attachments == null || attachments.size() == 0)
                throw new NSDException("Attachments not found in GetPackage response");

            DataHandler dataHandler = attachments.iterator().next().getDataHandler();
            try (FileOutputStream os = new FileOutputStream(new File(directoryOut, fileName))) {
                IOUtils.copyAndCloseInput(dataHandler.getInputStream(), os);
            }
            logger.debug("GetPackage finished. File " + fileName + " saved");
        } catch (WsFault e) {
            logger.error("GetPackage error", e);
            FaultInfo faultInfo = e.getFaultInfo();
            if (faultInfo != null) {
                throw new NSDException(faultInfo.getErrorCode(), faultInfo.getErrorDesc(), e);
            }
            throw e;
        }


    }


}
