package inist.nsdws.rest;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonSyntaxException;
import inist.nsdws.common.ClientBase;
import inist.nsdws.common.Crypter;
import inist.nsdws.common.NSDException;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PutMethod;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;

/**
 * Description:  NSD REST Web-service client
 * User: Jury Rib, Inist
 * Date: 21.11.13
 * Time: 14:27
 */
public class ClientREST extends ClientBase {
    public static final String CHARSET = "UTF-8";


    final Logger logger = LoggerFactory.getLogger(ClientREST.class);
    private final String serviceURI;
    private final String serviceVersion;
    private String memberCode;
    private Crypter crypter;

    /**
     * NSD REST Web-service client constructor
     *
     * @throws Exception
     */
    public ClientREST() throws Exception {
        serviceURI = properties.getProperty("rest.service.URI");
        if (serviceURI == null) throw new IllegalArgumentException("Не задан URI REST-сервиса");

        serviceVersion = properties.getProperty("rest.service.version", "1");

        //check NSD depo member code
        memberCode = properties.getProperty("nsd.memberCode");
        if (memberCode == null) throw new IllegalArgumentException("Не задан код участника НРД");

        //check keystore path
        String keystorePath = properties.getProperty("keystore.path");
        if (keystorePath == null) throw new IllegalArgumentException("Не задан путь к хранилищу ключей");

        //check keystore password
        String keystorePassword = properties.getProperty("keystore.password");
        if (keystorePassword == null) throw new IllegalArgumentException("Не задан пароль доступа к хранилищу ключей");

        //check private key alias
        String privateKeyAlias = properties.getProperty("privateKey.alias");
        if (privateKeyAlias == null)
            throw new IllegalArgumentException("Не задан код (alias) первичного ключа в хранилище");
        //check NSD key  certificate alias
        String nsdCertAlias = properties.getProperty("serverCert.alias");
        if (nsdCertAlias == null)
            throw new IllegalArgumentException("Не задан код (alias) сертификата сервера в хранилище");

        crypter = new Crypter(keystorePath, keystorePassword, privateKeyAlias, nsdCertAlias);
    }

    /**
     * retrieve current markups
     *
     * @return current markups
     * @throws Exception
     */
    public MarkupListBean getMarkups() throws Exception {
        //create part of request URL to sign
        String signedURLPart = "/v" + serviceVersion + "/participants/" + memberCode + "/markups";
        String fullURL = serviceURI + signedURLPart;

        //init http client
        HttpClient client = new HttpClient();
        HttpMethod method = new GetMethod(fullURL);
        method.addRequestHeader("Content-Type", "text/html; charset=UTF-8");
        //add client's certificate header
        String certificate = crypter.getCertificateBase64();
        method.addRequestHeader("X-CLIENTCERT", certificate);
        //add signature header
        String signature = crypter.makePKCS7(signedURLPart.getBytes(CHARSET));
        method.addRequestHeader("X-WSALAM-SIGN", signature);

        //send request
        MarkupListBean markups = null;
        try {
            logger.debug("getMarkup request " + fullURL);
            // Execute the method.
            int statusCode = client.executeMethod(method);
            logger.debug("getMarkup response status " + statusCode);
            // Read the response body.
            byte[] responseBody = method.getResponseBody();
            String result = responseBody == null ? null : new String(responseBody, CHARSET);
            logger.debug("getMarkup response body: " + result);
            if (statusCode == 200) {
                Gson gson = new GsonBuilder().setDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXXX").create();
                markups = gson.fromJson(result, MarkupListBean.class);
            } else {
                String errorMessage = null;
                //try read result
                if (result != null) {
                    errorMessage = parseErrorJson(result);
                }
                throw new NSDException(statusCode, errorMessage);
            }

        } catch (HttpException e) {
            logger.error("Fatal protocol violation", e);
            throw e;
        } catch (IOException e) {
            logger.error("Fatal transport error", e);
            throw e;
        } finally {
            // Release the connection.
            method.releaseConnection();
        }
        return markups;
    }

    private String parseErrorJson(String resultJson) {
        String error;
        try {
            Gson gson = new Gson();
            error = gson.fromJson(resultJson, String.class);
        } catch (JsonSyntaxException e) {
            error = resultJson;
        }
        return error;
    }


    /**
     * save markups to server
     *
     * @param markups new markups
     * @throws Exception
     */
    public void putMarkups(MarkupListBean markups) throws Exception {
        //create  json object
        Gson gson = new GsonBuilder().setDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXXX").create();
        String json = gson.toJson(markups);

        //create part of request URL to sign
        String signedURLPart = "/v" + serviceVersion + "/participants/" + memberCode + "/markups";

        String fullURL = serviceURI + signedURLPart;

        //init http client
        HttpClient client = new HttpClient();
        PutMethod method = new PutMethod(fullURL);
        method.setRequestHeader("Cache-Control", "no-cache");
        //add client's certificate header
        String certificate = crypter.getCertificateBase64();
        method.addRequestHeader("X-CLIENTCERT", certificate);
        //add signature header
        String signature = crypter.makePKCS7((signedURLPart + json).getBytes(CHARSET));
        method.addRequestHeader("X-WSALAM-SIGN", signature);

        //send request
        method.setRequestEntity(new StringRequestEntity(json, "application/json", CHARSET));
        try {
            logger.debug("postMarkup request " + fullURL);
            // Execute the method.
            int statusCode = client.executeMethod(method);
            logger.debug("postMarkup response status " + statusCode);
            // Read the response body.
            byte[] responseBody = method.getResponseBody();
            String result = responseBody == null ? null : new String(responseBody, CHARSET);
            logger.debug("postMarkup response body: " + result);
            if (statusCode != 200 && statusCode != 201) {
                String errorMessage = null;
                //try read result
                if (result != null) {
                    errorMessage = parseErrorJson(result);
                }
                throw new NSDException(statusCode, errorMessage);
            }

        } catch (HttpException e) {
            logger.error("Fatal protocol violation", e);
            throw e;
        } catch (IOException e) {
            logger.error("Fatal transport error", e);
            throw e;
        } finally {
            // Release the connection.
            method.releaseConnection();
        }
    }

}
