package inist.nsdws.common;

import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaCertStore;
import org.bouncycastle.cms.*;
import org.bouncycastle.cms.jcajce.JcaSignerInfoGeneratorBuilder;
import org.bouncycastle.cms.jcajce.JcaSimpleSignerInfoVerifierBuilder;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.operator.jcajce.JcaDigestCalculatorProviderBuilder;
import org.bouncycastle.util.Store;
import org.bouncycastle.util.encoders.Base64;

import java.io.FileInputStream;
import java.io.InputStream;
import java.security.*;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateCrtKey;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * Description:
 * User: Jury Rib
 * Date: 25.11.13
 * Time: 15:24
 */
public class Crypter {

    private RSAPrivateCrtKey privateKey;
    private X509Certificate pkCertificate;
    private X509Certificate nsdCertificate;
    private String base64Cert;

    /**
     * Crypter initialization
     *
     * @param keyStorePath    keyStore path
     * @param password        keyStore password
     * @param privateKeyAlias alias of private key certificate
     * @param nsdCertAlias    alias of nsd public key certificate
     * @throws CryptoException
     */
    public Crypter(String keyStorePath, String password, String privateKeyAlias, String nsdCertAlias) throws CryptoException {
        if (privateKeyAlias == null)
            throw new CryptoException("Private key alias not specified");
        KeyStore keyStore;
        try {
            keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
            try (InputStream fis = new FileInputStream(keyStorePath)) {
                keyStore.load(fis, password.toCharArray());
            }
            pkCertificate = (X509Certificate) keyStore.getCertificate(privateKeyAlias);
            byte[] certificateBody = pkCertificate.getEncoded();
            base64Cert = Base64.toBase64String(certificateBody);
            privateKey = (RSAPrivateCrtKey) keyStore.getKey(privateKeyAlias, password.toCharArray());
            nsdCertificate = (X509Certificate) keyStore.getCertificate(nsdCertAlias);
        } catch (Exception e) {
            throw new CryptoException("KeyStore init error", e);
        }

    }

    public String getCertificateBase64() {

        return base64Cert;

    }

    public RSAPrivateCrtKey getPrivateKey() {
        return privateKey;
    }

    public X509Certificate getCertificate() {
        return pkCertificate;
    }

    /**
     * create RSA-SHA1 signature
     *
     * @param body data to sign
     * @return base64-encoded signature
     * @throws GeneralSecurityException
     */
    public String rsaSignData(byte[] body) throws GeneralSecurityException {
        Signature sign = Signature.getInstance("Sha1WithRSA");
        sign.initSign(privateKey);
        sign.update(body);
        byte[] bb = sign.sign();
        return Base64.toBase64String(bb);
    }


    /**
     * create SHA1 digest
     *
     * @param body data bytes
     * @return base64-encoded digest
     * @throws NoSuchAlgorithmException
     */
    public String digestSHA1(byte[] body) throws NoSuchAlgorithmException {
        //org.apache.jcp.xml.dsig.internal.dom.DOMExcC14NMethod@4ee50f28
        MessageDigest md = MessageDigest.getInstance("SHA1");
        byte[] digest = md.digest(body);
        return Base64.toBase64String(digest);
    }

    /**
     * make PKCS7 packed SHA1RSA signature
     *
     * @param dataToSign data bytes
     * @return base64-encoded PKCS7 digest
     * @throws Exception
     */
    public String makePKCS7(byte[] dataToSign) throws Exception {
        List<X509Certificate> certList = new ArrayList<X509Certificate>(1);
        CMSTypedData msg = new CMSProcessableByteArray(dataToSign);

        certList.add(pkCertificate);

        Store certs = new JcaCertStore(certList);

        CMSSignedDataGenerator gen = new CMSSignedDataGenerator();
        ContentSigner sha1Signer = new JcaContentSignerBuilder("SHA1withRSA").setProvider("BC").build(privateKey);

        gen.addSignerInfoGenerator(
                new JcaSignerInfoGeneratorBuilder(
                        new JcaDigestCalculatorProviderBuilder().setProvider("BC").build())
                        .build(sha1Signer, pkCertificate));

        gen.addCertificates(certs);

        CMSSignedData sigData = gen.generate(msg, false);
        byte[] bb = sigData.getEncoded();
        return Base64.toBase64String(bb);
    }

    /**
     * verify PKCS7 packed SHA1RSA signature
     *
     * @param sigData data bytes
     * @throws Exception
     */
    public void verifyPKCS7(byte[] sigData) throws Exception {
        CMSSignedData s = new CMSSignedData(sigData);
        Store store = s.getCertificates();
        SignerInformationStore signers = s.getSignerInfos();
        Collection c = signers.getSigners();
        for (Object o : c) {
            SignerInformation signer = (SignerInformation) o;
            Collection certCollection = store.getMatches(signer.getSID());
            Iterator certIt = certCollection.iterator();
            X509CertificateHolder certHolder = (X509CertificateHolder) certIt.next();
            if (!certHolder.getSerialNumber().equals(nsdCertificate.getSerialNumber()))
                throw new NSDException("Ошибка проверки подписи: Сертификат в подписи (SN=" + certHolder.getSerialNumber() + ") не соответствует сертификату NSD(CN=" + nsdCertificate.getSerialNumber() + ") ");

            if (!signer.verify(new JcaSimpleSignerInfoVerifierBuilder().setProvider("BC").build(nsdCertificate))) {
                throw new NSDException("Подпись не верна");
            }
        }
    }

}
